/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL point attributes in RDB 2 database files
 * \version 2015-11-05/AW: Initial version
 * \version 2015-11-20/AW: Minimum value of "riegl.pulse_width" fixed
 * \version 2016-09-30/AW: New attributes for voxel and plane patch datasets
 * \version 2016-10-25/AW: Point coordinates correction vector attribute added
 * \version 2016-11-17/AW: Attributes for voxel datasets updated
 * \version 2016-11-23/AW: Added constants for named point attribute values
 * \version 2016-12-05/AW: Compression options added
 * \version 2016-12-20/AW: JSON: switched names from camel case to underscores
 * \version 2017-01-20/AW: Maximum value of "riegl.pulse_width" fixed
 * \version 2017-01-23/AW: New attributes for laser shots added: riegl.shot_...
 * \version 2017-01-31/AW: New attributes for FWA and MTA tools added
 * \version 2017-02-24/AW: Definition of riegl.covariances fixed
 * \version 2017-03-02/AW: New point attribute riegl.plane_count added
 * \version 2017-03-22/AW: Optional attribute scale factor added
 * \version 2017-05-10/AW: Definition of riegl.plane_count fixed
 * \version 2017-06-27/AW: Description of target_index and target_count fixed
 * \version 2017-08-22/AW: New attributes for waveform sample blocks and values
 * \version 2017-10-03/AW: Description of wfm_sbl_first and wfm_sda_first fixed
 * \version 2017-10-09/AW: Definition of riegl.amplitude modified
 * \version 2017-10-24/AW: New attribute riegl.line_angle_coarse added
 * \version 2017-11-10/AW: Definition of riegl.timestamp modified (#2588)
 * \version 2017-11-20/AW: Resolution of riegl.pca_extents increased
 * \version 2017-11-21/AW: Attributes for trajectory (position+orientation) files
 * \version 2017-11-22/AW: New attribute riegl.voxel_collapsed added
 * \version 2017-11-22/AW: Resolution of riegl.std_dev increased
 * \version 2018-01-10/AW: New attribute riegl.direction_coarse added
 * \version 2018-01-15/AW: Comments of riegl.fwa updated
 * \version 2018-02-14/AW: Definition of riegl.timestamp fixed (#2588)
 * \version 2018-02-22/AW: Description of target_index and target_count fixed
 * \version 2018-03-09/AW: New attribute property "invalid value" added (#3047)
 * \version 2018-03-26/AW: Definition of riegl.temperature modified
 * \version 2018-04-20/AW: Definition of riegl.[pof_]timestamp fixed (#2588)
 * \version 2018-05-24/AW: Attributes for voxel and pixel datasets added
 * \version 2018-06-08/AW: Attributes for waveform and echo datasets added
 * \version 2018-06-25/AW: Definition of riegl.wfm_sbl_std_dev fixed
 * \version 2018-06-28/AW: New attributes riegl.pps_timestamp_[ex|in]tern added
 * \version 2018-07-04/AW: Invalid value for "riegl.pulse_width" added
 * \version 2018-11-19/AW: New attribute riegl.direction_medium added
 * \version 2018-11-28/AW: Definition of riegl.scan_angle modified
 * \version 2018-12-04/AW: Definition of riegl.plane_spread removed
 * \version 2019-02-25/AW: New attribute riegl.xyz_socs added
 * \version 2019-03-14/AW: New attribute riegl.voxel_index added
 * \version 2019-03-14/AW: New attribute riegl.voxel_linear_sums added
 * \version 2019-03-14/AW: New attribute riegl.voxel_square_sums added
 * \version 2019-03-14/AW: New attribute riegl.pixel_linear_sums added
 * \version 2019-03-14/AW: New attribute riegl.pixel_square_sums added
 * \version 2019-03-14/AW: New attribute riegl.hydro_refraction_corrected added
 * \version 2019-03-14/AW: New attribute riegl.hydro_intersection_point added
 * \version 2019-03-14/AW: New attribute riegl.hydro_intersection_normal added
 * \version 2019-04-10/AW: Definition of riegl.xy_map and riegl.xyz_map fixed
 * \version 2019-04-15/AW: Attributes were separated into groups
 * \version 2019-05-16/AW: New attribute riegl.hydro_wsm_uncertainty added
 * \version 2019-05-22/AW: New attribute riegl.plane_patch_distance added
 * \version 2019-05-22/AW: New attribute riegl.plane_patch_angular_distance added
 * \version 2019-05-22/AW: Titles of riegl.pof_satellites_* attributes fixed
 * \version 2019-06-25/AW: New attribute riegl.pof_satellites_qzss added
 * \version 2019-11-07/AW: Attributes riegl.pof_xyz and riegl.pof_*_ned added
 * \version 2019-11-11/AW: Attribute riegl.plane_patch_lateral_distance added
 * \version 2019-11-15/AW: Attribute riegl.plane_patch_link_vector added
 * \version 2019-11-22/AW: Attribute riegl.plane_occupancy added
 * \version 2019-11-22/AW: Attributes riegl.raw_(line|frame)_angle added
 * \version 2019-11-25/AW: Attribute riegl.plane_cog_link added
 * \version 2019-11-25/AW: Attribute riegl.plane_confidence_normal added
 * \version 2019-12-02/AW: Attribute riegl.match_count added
 * \version 2019-12-13/AW: Attributes for tie-/control objects added
 * \version 2019-12-19/AW: Attributes for tie-/control objects added
 * \version 2019-12-19/AW: Title and description of riegl.mta_unresolved updated
 * \version 2020-01-08/AW: Move riegl.height_center/mean/min/max before riegl.point_count
 * \version 2020-04-15/AW: Attribute riegl.reference_object_id updated
 * \version 2020-06-12/AW: Attribute riegl.scanner_position added
 * \version 2020-09-10/AW: Attribute riegl.line_angle_reduced added
 * \version 2020-09-17/AW: Attribute riegl.background_radiation updated
 * \version 2020-09-24/AW: Attribute riegl.background_radiation_si added
 * \version 2020-09-24/AW: Attribute riegl.background_radiation_ingaas added
 * \version 2020-09-24/AW: Attribute riegl.temperature_estimated_si added
 * \version 2020-09-24/AW: Attribute riegl.temperature_estimated_ingaas added
 * \version 2020-09-25/AW: Attribute riegl.temperature_estimated_ingaas_si added
 * \version 2020-10-08/AW: Attribute riegl.window_echo_impact_corrected added
 * \version 2020-11-16/AW: Attribute riegl.point_count_grid_cell added (#3720)
 * \version 2020-11-26/AW: Resolution of riegl.pof_accuracy_* attributes updated (#3761)
 * \version 2020-12-02/AW: Move riegl.pof_xyz after riegl.pof_yaw (#3760)
 * \version 2020-12-03/AW: Add riegl.scan_line_index and riegl.shot_index_line (#3759)
 * \version 2021-02-02/AW: Attribute riegl.mirror_facet updated (storage "variable")
 * \version 2021-02-02/AW: Attribute riegl.plane_slope_class added (rdbplanes/#7)
 * \version 2021-02-16/AW: Attribute riegl.source_cloud_count added (#3810)
 * \version 2021-03-03/AW: Attribute riegl.shot_direction_levelled added (#3820)
 * \version 2021-04-14/AW: Attribute riegl.obs_confidence_xy added (#3861)
 * \version 2021-04-14/AW: Attribute riegl.obs_confidence_z added (#3861)
 * \version 2021-04-14/AW: Attribute riegl.obs_signal_confidence_rot added (#3861)
 * \version 2021-06-30/AW: Attributes riegl.pca_axis_min and _max updated (#3930)
 * \version 2021-07-09/AW: Attribute riegl.extinction added (#3935)
 * \version 2021-07-14/AW: Attributes riegl.svb_* added (#3945)
 * \version 2021-09-08/AW: Attribute riegl.dynamic_object_point added (#3979)
 * \version 2021-09-08/AW: Attribute riegl.single_source_point added (#3975)
 * \version 2021-09-08/AW: Attribute riegl.mirror_object_point added (#3978)
 * \version 2021-09-20/AW: Attribute riegl.plane_cluster_id added (#3997)
 * \version 2021-11-04/AW: Attribute riegl.nir added (#4042)
 * \version 2021-12-07/AW: Add riegl.obs_confidence_range/_theta/_phi (#4075)
 * \version 2022-01-26/AW: Add optional point attribute tags (#4128)
 * \version 2022-03-11/AW: Add "boolean" and "enumeration" tags (#4128)
 * \version 2022-03-18/AW: Add named values for enumeration attributes (#4128)
 * \version 2022-03-31/AW: Attribute riegl.source_indicator updated (#4128/17)
 * \version 2022-04-13/AW: Attribute riegl.target_type added (#4188)
 * \version 2022-04-20/AW: Attribute riegl.shot_biaxial_shift added (#4195)
 * \version 2022-05-20/AW: Attribute riegl.cp_search_radius added (#4236)
 * \version 2022-05-20/AW: Attribute riegl.cp_maximum_distance added (#4236)
 * \version 2022-05-30/AW: Attribute riegl.dyntrig_uncertain_point added (#4217)
 * \version 2022-08-16/AW: Attribute riegl.segment_id added (#4307)
 * \version 2022-09-06/AW: Attribute renamed to riegl.mta_uncertain_point (#4335)
 * \version 2022-10-06/AW: Attribute riegl.rgba updated (description)
 * \version 2022-10-10/AW: Add "NED" suffix to the titles of "riegl.pof_*_ned"
 * \version 2022-11-09/AW: Attribute riegl.pof_age_of_corrections added (#4412)
 * \version 2022-11-09/AW: Attribute riegl.pof_baseline_length added (#4412)
 * \version 2022-11-09/AW: Attribute riegl.pof_solution_gnss added (#4412)
 * \version 2023-01-19/AW: Attribute riegl.event_timestamp added (#4449)
 * \version 2023-01-19/AW: Attribute riegl.event_index added (#4449)
 * \version 2023-01-19/AW: Attribute riegl.event_pulse_width added (#4449)
 * \version 2023-01-19/AW: Attribute riegl.event_fake_pulse added (#4449)
 * \version 2023-01-30/AW: Attribute riegl.accelerometer updated (#4523)
 * \version 2023-01-30/AW: Attribute riegl.gyroscope updated (#4523)
 * \version 2023-04-21/AW: Attribute riegl.magnetic_field_sensor updated (#4603)
 * \version 2023-04-21/AW: Attribute riegl.distance_to_surface updated (#4612)
 * \version 2023-05-24/AW: Attribute riegl.point_vector added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.pca_diameter added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.angle_of_incidence added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.eigenvalue_norm_min added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.eigenvalue_norm_max added (#4642)
 * \version 2023-05-30/AW: Add "timestamp" tag for timestamp attributes (#4433)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_axis_direction added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_radius added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_height added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_origin_confidence added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_radius_confidence added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_direction_confidence added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_inlier_count added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_outlier_count added (#4648)
 * \version 2023-06-02/AW: Attribute riegl.point_inside_aabb_bocs added (#4653)
 * \version 2023-06-02/AW: Attribute riegl.point_outside_aabb_bocs added (#4653)
 * \version 2023-06-07/AW: Update resolutions of direction vector attributes (#4669)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_min added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_max added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_mean added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_median added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_std added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_mad added (#4673)
 * \version 2023-06-15/AW: Attribute riegl.accelerometer updated (#4689)
 * \version 2023-06-15/AW: Attribute riegl.gyroscope updated (#4689)
 * \version 2023-11-13/AW: Attribute riegl.svb_reflectance_volumetric added (#4819)
 * \version 2023-11-13/AW: Attribute riegl.svb_reflectance_bottom_uncorrected added (#4819)
 * \version 2023-11-13/AW: Attribute riegl.svb_attenuation_volumetric added (#4819)
 * \version 2023-11-13/AW: Attribute riegl.svb_target_attenuation_coefficient_volumetric added (#4819)
 * \version 2023-12-06/AW: Update invalid value of riegl.amplitude (#4870)
 * \version 2023-12-06/AW: Update invalid value of riegl.reflectance (#4870)
 * \version 2023-12-11/AW: Update invalid value of riegl.background_radiation_si (#4873)
 * \version 2023-12-11/AW: Update invalid value of riegl.background_radiation_ingaas (#4873)
 * \version 2024-01-30/AW: Attribute riegl.xyz_lvcs added (#4929)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_north_east added (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_east_down added (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_down_north added (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_north updated (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_east updated (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_down updated (#5076)
 * \version 2024-05-16/AW: Attribute riegl.tpu_wsm_z_uncertainty added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_wsm_normal_uncertainty added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_xyz_uncertainties added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_thu added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_tvu added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.xyz_map_min added (#5117)
 * \version 2024-05-16/AW: Attribute riegl.xyz_map_max added (#5117)
 * \version 2024-10-09/AW: Attribute riegl.cp_surface_northing_angle added (#5327)
 * \version 2024-10-09/AW: Attribute riegl.cp_surface_northing_tolerance_angle added (#5327)
 * \version 2024-10-15/AW: Attribute riegl.pof_xyz_velocity added (#5333)
 * \version 2024-10-15/AW: Attribute riegl.pof_xyz_lvcs_velocity added (#5333)
 * \version 2024-10-15/AW: Attribute riegl.pof_yaw_velocity added (#5333)
 * \version 2024-10-16/AW: Update titles to use "Title Case" style (#5335)
 * \version 2024-10-21/AW: Attribute riegl.pof_roll_velocity added (#5342)
 * \version 2024-10-21/AW: Attribute riegl.pof_pitch_velocity added (#5342)
 * \version 2024-11-15/AW: Attribute riegl.accelerometer_bias added (#5385)
 * \version 2024-11-15/AW: Attribute riegl.accelerometer_scale added (#5385)
 * \version 2024-11-15/AW: Attribute riegl.gyroscope_bias added (#5385)
 * \version 2024-11-15/AW: Attribute riegl.gyroscope_scale added (#5385)
 * \version 2024-12-11/AW: Attribute riegl.pof_xyz_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.pof_roll_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.pof_pitch_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.pof_yaw_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.mta_uncertain_point_vol added (#5357)
 * \version 2024-12-11/AW: Attribute riegl.mta_uncertain_point_vol_diff added (#5357)
 * \version 2025-03-21/AW: Attribute riegl.dmi_count added (#5571)
 * \version 2025-03-21/AW: Attribute riegl.dmi_path_length_raw added (#5571)
 * \version 2025-03-21/AW: Attribute riegl.pof_course added (#5595)
 * \version 2025-03-21/AW: Attribute riegl.pof_xyz_ned_velocity added (#5595)
 * \version 2025-03-24/AW: Attribute riegl.dmi_scale added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_ned added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_north added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_east added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_down added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_north_east added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_east_down added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_down_north added (#5595)
 * \version 2025-03-31/AW: Update description of riegl.pof_xyz_velocity (#5595)
 * \version 2025-03-31/AW: Update description of riegl.pof_xyz_ned_velocity (#5595)
 * \version 2025-03-31/AW: Update description of riegl.pof_xyz_lvcs_velocity (#5595)
 * \version 2025-04-03/AW: Attribute riegl.pof_undulation added (#5595)
 * \version 2025-08-29/AW: Attribute riegl.obs_confidence_xyz added (#5864)
 *
 *******************************************************************************
 */

#ifndef RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B0
#define RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B0

#include "math.h"
#include "stdlib.h"
#include "stdint.h"
#include "riegl/rdb.h"

// Point attribute definition version
const char* const RDB_POINT_ATTRIBUTES_VERSION = "1.4.8";
const char* const RDB_POINT_ATTRIBUTES_DATE = "2025-08-29";

// Invalid value
const double RDB_NO_INVALID_VALUE = NAN; // attribute has no invalid value

// Storage classes
const unsigned char RDB_STORAGE_CONSTANT = 1; // value cannot be changed
const unsigned char RDB_STORAGE_VARIABLE = 2; // value can change from time to time
const unsigned char RDB_STORAGE_DYNAMIC = 3; // value is likely to be changed often

// Compression options
const unsigned char RDB_COMPRESSION_DEFAULT = 0; // nothing special, just use default compression algorithm
const unsigned char RDB_COMPRESSION_DELTA = 1; // calculate differences between two consecutive values
const unsigned char RDB_COMPRESSION_SHUFFLE = 2; // shuffle bytes of point attribute values
const unsigned char RDB_COMPRESSION_DELTA_SHUFFLE = 3; // calculate differences and shuffle bytes

// Container record for point attribute details
typedef struct PointAttributeInfo
{
    const char* name;
    const char* title;
    const char* tags;
    const char* description;
    const char* unitSymbol;
    uint32_t    length;
    double      resolution;
    double      minimumValue;
    double      maximumValue;
    double      defaultValue;
    double      invalidValue;
    const char* namedValues;
    uint8_t     storageClass;
    uint8_t     compressionOptions;
    const char* lodSettings;
    double      scaleFactor;
} PointAttributeInfo;

// Copy point attribute definition
static bool rdb_assign_point_attribute(
    RDBContext                  *context,
    RDBPointcloudPointAttribute *target,
    const PointAttributeInfo    *source
)
{
    return (rdb_pointcloud_point_attribute_set_name               (context, target, source->name)               == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_title              (context, target, source->title)              == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_tags               (context, target, source->tags)               == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_description        (context, target, source->description)        == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_unit_symbol        (context, target, source->unitSymbol)         == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_length_u32         (context, target, source->length)             == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_resolution         (context, target, source->resolution)         == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_minimum_value      (context, target, source->minimumValue)       == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_maximum_value      (context, target, source->maximumValue)       == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_default_value      (context, target, source->defaultValue)       == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_invalid_value      (context, target, source->invalidValue)       == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_named_values       (context, target, source->namedValues)        == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_storage_class      (context, target, source->storageClass)       == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_compression_options(context, target, source->compressionOptions) == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_lod_settings       (context, target, source->lodSettings)        == RDB_SUCCESS)
        && (rdb_pointcloud_point_attribute_set_scale_factor       (context, target, source->scaleFactor)        == RDB_SUCCESS);
}

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Coordinates/Vectors"
//______________________________________________________________________________
//

// Cartesian point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_XYZ = {
    /* name               */ "riegl.xyz",
    /* title              */ "XYZ",
    /* tags               */ "position, transform",
    /* description        */ "Cartesian point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -535000.0,
    /* maximumValue       */ 535000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_XYZ_X = 0;
const size_t RDB_RIEGL_XYZ_Y = 1;
const size_t RDB_RIEGL_XYZ_Z = 2;

// Cartesian point coordinates wrt. scanner's coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_XYZ_SOCS = {
    /* name               */ "riegl.xyz_socs",
    /* title              */ "XYZ SOCS",
    /* tags               */ "position",
    /* description        */ "Cartesian point coordinates wrt. scanner's coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -535000.0,
    /* maximumValue       */ 535000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_XYZ_SOCS_X = 0;
const size_t RDB_RIEGL_XYZ_SOCS_Y = 1;
const size_t RDB_RIEGL_XYZ_SOCS_Z = 2;

// Cartesian point coordinates wrt. levelled vehicle coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_XYZ_LVCS = {
    /* name               */ "riegl.xyz_lvcs",
    /* title              */ "XYZ LVCS",
    /* tags               */ "position",
    /* description        */ "Cartesian point coordinates wrt. levelled vehicle coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.1,
    /* minimumValue       */ -10000.0,
    /* maximumValue       */ 10000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_XYZ_LVCS_X = 0;
const size_t RDB_RIEGL_XYZ_LVCS_Y = 1;
const size_t RDB_RIEGL_XYZ_LVCS_Z = 2;

// Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, 0: Easting, 1: Northing, 2: Height)
const PointAttributeInfo RDB_RIEGL_XYZ_MAP = {
    /* name               */ "riegl.xyz_map",
    /* title              */ "XYZ Map",
    /* tags               */ "position, transform",
    /* description        */ "Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, 0: Easting, 1: Northing, 2: Height)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -20037508.343,
    /* maximumValue       */ 20037508.343,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_XYZ_MAP_EASTING = 0;
const size_t RDB_RIEGL_XYZ_MAP_NORTHING = 1;
const size_t RDB_RIEGL_XYZ_MAP_HEIGHT = 2;

// Minimum distances to the voxel corner (0: dx, 1: dy, 2: dz)
const PointAttributeInfo RDB_RIEGL_XYZ_MAP_MIN = {
    /* name               */ "riegl.xyz_map_min",
    /* title              */ "XYZ Map Minimum",
    /* tags               */ "direction, transform",
    /* description        */ "Minimum distances to the voxel corner (0: dx, 1: dy, 2: dz)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Maximum distances to the voxel corner (0: dx, 1: dy, 2: dz)
const PointAttributeInfo RDB_RIEGL_XYZ_MAP_MAX = {
    /* name               */ "riegl.xyz_map_max",
    /* title              */ "XYZ Map Maximum",
    /* tags               */ "direction, transform",
    /* description        */ "Maximum distances to the voxel corner (0: dx, 1: dy, 2: dz)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, 0: Easting, 1: Northing)
const PointAttributeInfo RDB_RIEGL_XY_MAP = {
    /* name               */ "riegl.xy_map",
    /* title              */ "XY Map",
    /* tags               */ "position, transform",
    /* description        */ "Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, 0: Easting, 1: Northing)",
    /* unitSymbol         */ "m",
    /* length             */ 2,
    /* resolution         */ 0.00933069192934280443318950659659094526432454586029052734375,
    /* minimumValue       */ -20037508.3427892439067363739013671875,
    /* maximumValue       */ 20037508.333458550274372100830078125,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_XY_MAP_EASTING = 0;
const size_t RDB_RIEGL_XY_MAP_NORTHING = 1;

// Corrections that were applied (added) to the Cartesian point coordinates (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_XYZ_CORRECTIONS = {
    /* name               */ "riegl.xyz_corrections",
    /* title              */ "XYZ Corrections",
    /* tags               */ "direction, transform",
    /* description        */ "Corrections that were applied (added) to the Cartesian point coordinates (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -5000.0,
    /* maximumValue       */ 5000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_XYZ_CORRECTIONS_X = 0;
const size_t RDB_RIEGL_XYZ_CORRECTIONS_Y = 1;
const size_t RDB_RIEGL_XYZ_CORRECTIONS_Z = 2;

// Target distance wrt. SOCS origin
const PointAttributeInfo RDB_RIEGL_RANGE = {
    /* name               */ "riegl.range",
    /* title              */ "Range",
    /* tags               */ "",
    /* description        */ "Target distance wrt. SOCS origin",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Polar angle (inclination) wrt. SOCS (0..180°)
const PointAttributeInfo RDB_RIEGL_THETA = {
    /* name               */ "riegl.theta",
    /* title              */ "Theta",
    /* tags               */ "",
    /* description        */ "Polar angle (inclination) wrt. SOCS (0..180°)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Azimuthal angle wrt. SOCS (0..360°)
const PointAttributeInfo RDB_RIEGL_PHI = {
    /* name               */ "riegl.phi",
    /* title              */ "Phi",
    /* tags               */ "",
    /* description        */ "Azimuthal angle wrt. SOCS (0..360°)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Cartesian coordinates of the scanner position wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_SCANNER_POSITION = {
    /* name               */ "riegl.scanner_position",
    /* title              */ "Scanner Position",
    /* tags               */ "position, transform",
    /* description        */ "Cartesian coordinates of the scanner position wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -535000.0,
    /* maximumValue       */ 535000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_DIRECTION = {
    /* name               */ "riegl.direction",
    /* title              */ "Direction",
    /* tags               */ "direction, transform",
    /* description        */ "Laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_DIRECTION_X = 0;
const size_t RDB_RIEGL_DIRECTION_Y = 1;
const size_t RDB_RIEGL_DIRECTION_Z = 2;

// Laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_DIRECTION_MEDIUM = {
    /* name               */ "riegl.direction_medium",
    /* title              */ "Direction",
    /* tags               */ "direction, transform",
    /* description        */ "Laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 0.0007,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_DIRECTION_MEDIUM_X = 0;
const size_t RDB_RIEGL_DIRECTION_MEDIUM_Y = 1;
const size_t RDB_RIEGL_DIRECTION_MEDIUM_Z = 2;

// Coarse laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_DIRECTION_COARSE = {
    /* name               */ "riegl.direction_coarse",
    /* title              */ "Direction",
    /* tags               */ "direction, transform",
    /* description        */ "Coarse laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 0.015,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_DIRECTION_COARSE_X = 0;
const size_t RDB_RIEGL_DIRECTION_COARSE_Y = 1;
const size_t RDB_RIEGL_DIRECTION_COARSE_Z = 2;

// Laser beam origin wrt. SOCS (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_SHOT_ORIGIN = {
    /* name               */ "riegl.shot_origin",
    /* title              */ "Laser Shot Origin",
    /* tags               */ "position",
    /* description        */ "Laser beam origin wrt. SOCS (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 250.0e-6,
    /* minimumValue       */ -8.0,
    /* maximumValue       */ 8.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_SHOT_ORIGIN_X = 0;
const size_t RDB_RIEGL_SHOT_ORIGIN_Y = 1;
const size_t RDB_RIEGL_SHOT_ORIGIN_Z = 2;

// Laser beam biaxial shift vector wrt. SOCS (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_SHOT_BIAXIAL_SHIFT = {
    /* name               */ "riegl.shot_biaxial_shift",
    /* title              */ "Laser Shot Biaxial Shift",
    /* tags               */ "direction",
    /* description        */ "Laser beam biaxial shift vector wrt. SOCS (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 250.0e-6,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_SHOT_BIAXIAL_SHIFT_X = 0;
const size_t RDB_RIEGL_SHOT_BIAXIAL_SHIFT_Y = 1;
const size_t RDB_RIEGL_SHOT_BIAXIAL_SHIFT_Z = 2;

// Laser beam direction vector wrt. SOCS (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_SHOT_DIRECTION = {
    /* name               */ "riegl.shot_direction",
    /* title              */ "Laser Shot Direction",
    /* tags               */ "direction",
    /* description        */ "Laser beam direction vector wrt. SOCS (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 250.0e-9,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_SHOT_DIRECTION_X = 0;
const size_t RDB_RIEGL_SHOT_DIRECTION_Y = 1;
const size_t RDB_RIEGL_SHOT_DIRECTION_Z = 2;

// Laser beam direction vector wrt. ROCS (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_SHOT_DIRECTION_LEVELLED = {
    /* name               */ "riegl.shot_direction_levelled",
    /* title              */ "Laser Shot Direction Levelled",
    /* tags               */ "direction, transform",
    /* description        */ "Laser beam direction vector wrt. ROCS (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 250.0e-9,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_SHOT_DIRECTION_LEVELLED_X = 0;
const size_t RDB_RIEGL_SHOT_DIRECTION_LEVELLED_Y = 1;
const size_t RDB_RIEGL_SHOT_DIRECTION_LEVELLED_Z = 2;

// Target surface normal vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_SURFACE_NORMAL = {
    /* name               */ "riegl.surface_normal",
    /* title              */ "Surface Normal",
    /* tags               */ "direction, transform",
    /* description        */ "Target surface normal vector wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_SURFACE_NORMAL_X = 0;
const size_t RDB_RIEGL_SURFACE_NORMAL_Y = 1;
const size_t RDB_RIEGL_SURFACE_NORMAL_Z = 2;

// Vector derived from PCA on neighborhood: normal vector for planar and volumetric neighborhoods, direction vector for linear neighborhoods
const PointAttributeInfo RDB_RIEGL_POINT_VECTOR = {
    /* name               */ "riegl.point_vector",
    /* title              */ "Point Vector",
    /* tags               */ "direction, transform",
    /* description        */ "Vector derived from PCA on neighborhood: normal vector for planar and volumetric neighborhoods, direction vector for linear neighborhoods",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 0.0005,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_POINT_VECTOR_X = 0;
const size_t RDB_RIEGL_POINT_VECTOR_Y = 1;
const size_t RDB_RIEGL_POINT_VECTOR_Z = 2;

// Direction vector of shorter edge of plane patch wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_PLANE_UP = {
    /* name               */ "riegl.plane_up",
    /* title              */ "Plane Patch Up Vector",
    /* tags               */ "direction, transform",
    /* description        */ "Direction vector of shorter edge of plane patch wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PLANE_UP_X = 0;
const size_t RDB_RIEGL_PLANE_UP_Y = 1;
const size_t RDB_RIEGL_PLANE_UP_Z = 2;

// Vector connecting the center point of a plane patch with its center of gravity (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_PLANE_COG_LINK = {
    /* name               */ "riegl.plane_cog_link",
    /* title              */ "Plane COG Link Vector",
    /* tags               */ "direction, transform",
    /* description        */ "Vector connecting the center point of a plane patch with its center of gravity (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -5000.0,
    /* maximumValue       */ 5000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PLANE_COG_LINK_X = 0;
const size_t RDB_RIEGL_PLANE_COG_LINK_Y = 1;
const size_t RDB_RIEGL_PLANE_COG_LINK_Z = 2;

// Vector interconnecting the centers (riegl.xyz) of two matched plane patches (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_PLANE_PATCH_LINK_VECTOR = {
    /* name               */ "riegl.plane_patch_link_vector",
    /* title              */ "Plane Patch Link Vector",
    /* tags               */ "direction, transform",
    /* description        */ "Vector interconnecting the centers (riegl.xyz) of two matched plane patches (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -5000.0,
    /* maximumValue       */ 5000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PLANE_PATCH_LINK_VECTOR_X = 0;
const size_t RDB_RIEGL_PLANE_PATCH_LINK_VECTOR_Y = 1;
const size_t RDB_RIEGL_PLANE_PATCH_LINK_VECTOR_Z = 2;

// The eigenvector that belongs to the smallest eigenvalue (result of PCA, 0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_PCA_AXIS_MIN = {
    /* name               */ "riegl.pca_axis_min",
    /* title              */ "PCA Axis Minimum",
    /* tags               */ "direction, transform",
    /* description        */ "The eigenvector that belongs to the smallest eigenvalue (result of PCA, 0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PCA_AXIS_MIN_X = 0;
const size_t RDB_RIEGL_PCA_AXIS_MIN_Y = 1;
const size_t RDB_RIEGL_PCA_AXIS_MIN_Z = 2;

// The eigenvector that belongs to the greatest eigenvalue (result of PCA, 0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_PCA_AXIS_MAX = {
    /* name               */ "riegl.pca_axis_max",
    /* title              */ "PCA Axis Maximum",
    /* tags               */ "direction, transform",
    /* description        */ "The eigenvector that belongs to the greatest eigenvalue (result of PCA, 0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PCA_AXIS_MAX_X = 0;
const size_t RDB_RIEGL_PCA_AXIS_MAX_Y = 1;
const size_t RDB_RIEGL_PCA_AXIS_MAX_Z = 2;

// Direction of x-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_MODEL_CS_AXIS_X = {
    /* name               */ "riegl.model_cs_axis_x",
    /* title              */ "Model CS X-Axis",
    /* tags               */ "direction, transform",
    /* description        */ "Direction of x-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_MODEL_CS_AXIS_X_X = 0;
const size_t RDB_RIEGL_MODEL_CS_AXIS_X_Y = 1;
const size_t RDB_RIEGL_MODEL_CS_AXIS_X_Z = 2;

// Direction of y-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_MODEL_CS_AXIS_Y = {
    /* name               */ "riegl.model_cs_axis_y",
    /* title              */ "Model CS Y-Axis",
    /* tags               */ "direction, transform",
    /* description        */ "Direction of y-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_MODEL_CS_AXIS_Y_X = 0;
const size_t RDB_RIEGL_MODEL_CS_AXIS_Y_Y = 1;
const size_t RDB_RIEGL_MODEL_CS_AXIS_Y_Z = 2;

// Direction of z-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_MODEL_CS_AXIS_Z = {
    /* name               */ "riegl.model_cs_axis_z",
    /* title              */ "Model CS Z-Axis",
    /* tags               */ "direction, transform",
    /* description        */ "Direction of z-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_MODEL_CS_AXIS_Z_X = 0;
const size_t RDB_RIEGL_MODEL_CS_AXIS_Z_Y = 1;
const size_t RDB_RIEGL_MODEL_CS_AXIS_Z_Z = 2;

// Normalized cylinder axis direction vector
const PointAttributeInfo RDB_RIEGL_CYLINDER_AXIS_DIRECTION = {
    /* name               */ "riegl.cylinder_axis_direction",
    /* title              */ "Cylinder Axis Direction",
    /* tags               */ "direction, transform",
    /* description        */ "Normalized cylinder axis direction vector",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Processed values of an accelerometer
const PointAttributeInfo RDB_RIEGL_ACCELEROMETER = {
    /* name               */ "riegl.accelerometer",
    /* title              */ "Accelerometer Values",
    /* tags               */ "direction",
    /* description        */ "Processed values of an accelerometer",
    /* unitSymbol         */ "m/s²",
    /* length             */ 3,
    /* resolution         */ 5.0e-6,
    /* minimumValue       */ -300.0,
    /* maximumValue       */ 300.0,
    /* defaultValue       */ -300.0,
    /* invalidValue       */ -300.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Raw measurement values of an accelerometer (unit see metadata riegl.pose_sensors)
const PointAttributeInfo RDB_RIEGL_ACCELEROMETER_RAW = {
    /* name               */ "riegl.accelerometer_raw",
    /* title              */ "Accelerometer Raw Values",
    /* tags               */ "direction",
    /* description        */ "Raw measurement values of an accelerometer (unit see metadata riegl.pose_sensors)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 1.0,
    /* minimumValue       */ -32768.0,
    /* maximumValue       */ 32767.0,
    /* defaultValue       */ -32768.0,
    /* invalidValue       */ -32768.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated bias of accelerometer
const PointAttributeInfo RDB_RIEGL_ACCELEROMETER_BIAS = {
    /* name               */ "riegl.accelerometer_bias",
    /* title              */ "Accelerometer Bias",
    /* tags               */ "",
    /* description        */ "Estimated bias of accelerometer",
    /* unitSymbol         */ "m/s²",
    /* length             */ 3,
    /* resolution         */ 1e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated deviation of scale of accelerometer from 1
const PointAttributeInfo RDB_RIEGL_ACCELEROMETER_SCALE = {
    /* name               */ "riegl.accelerometer_scale",
    /* title              */ "Accelerometer Scale",
    /* tags               */ "",
    /* description        */ "Estimated deviation of scale of accelerometer from 1",
    /* unitSymbol         */ "1",
    /* length             */ 3,
    /* resolution         */ 1e-06,
    /* minimumValue       */ -0.1,
    /* maximumValue       */ 0.1,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -0.1,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Processed values of a gyroscope
const PointAttributeInfo RDB_RIEGL_GYROSCOPE = {
    /* name               */ "riegl.gyroscope",
    /* title              */ "Gyroscope Values",
    /* tags               */ "direction",
    /* description        */ "Processed values of a gyroscope",
    /* unitSymbol         */ "deg/s",
    /* length             */ 3,
    /* resolution         */ 1.0e-7,
    /* minimumValue       */ -2000.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ -2000.0,
    /* invalidValue       */ -2000.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Raw measurement values of a gyroscope (unit see metadata riegl.pose_sensors)
const PointAttributeInfo RDB_RIEGL_GYROSCOPE_RAW = {
    /* name               */ "riegl.gyroscope_raw",
    /* title              */ "Gyroscope Raw Values",
    /* tags               */ "direction",
    /* description        */ "Raw measurement values of a gyroscope (unit see metadata riegl.pose_sensors)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 1.0,
    /* minimumValue       */ -32768.0,
    /* maximumValue       */ 32767.0,
    /* defaultValue       */ -32768.0,
    /* invalidValue       */ -32768.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated bias of gyroscope
const PointAttributeInfo RDB_RIEGL_GYROSCOPE_BIAS = {
    /* name               */ "riegl.gyroscope_bias",
    /* title              */ "Gyroscope Bias",
    /* tags               */ "",
    /* description        */ "Estimated bias of gyroscope",
    /* unitSymbol         */ "deg/s",
    /* length             */ 3,
    /* resolution         */ 1e-06,
    /* minimumValue       */ -10.0,
    /* maximumValue       */ 10.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -10.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated deviation of scale of gyroscope from 1
const PointAttributeInfo RDB_RIEGL_GYROSCOPE_SCALE = {
    /* name               */ "riegl.gyroscope_scale",
    /* title              */ "Gyroscope Scale",
    /* tags               */ "",
    /* description        */ "Estimated deviation of scale of gyroscope from 1",
    /* unitSymbol         */ "1",
    /* length             */ 3,
    /* resolution         */ 1e-06,
    /* minimumValue       */ -0.1,
    /* maximumValue       */ 0.1,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -0.1,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Processed values of a magnetic field sensor
const PointAttributeInfo RDB_RIEGL_MAGNETIC_FIELD_SENSOR = {
    /* name               */ "riegl.magnetic_field_sensor",
    /* title              */ "Magnetic Field Sensor Values",
    /* tags               */ "direction",
    /* description        */ "Processed values of a magnetic field sensor",
    /* unitSymbol         */ "µT",
    /* length             */ 3,
    /* resolution         */ 0.01,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ -1000.0,
    /* invalidValue       */ -1000.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Raw measurement values of a magnetic field sensor (unit see metadata riegl.pose_sensors)
const PointAttributeInfo RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW = {
    /* name               */ "riegl.magnetic_field_sensor_raw",
    /* title              */ "Magnetic Field Sensor Raw Values",
    /* tags               */ "direction",
    /* description        */ "Raw measurement values of a magnetic field sensor (unit see metadata riegl.pose_sensors)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 1.0,
    /* minimumValue       */ -32768.0,
    /* maximumValue       */ 32767.0,
    /* defaultValue       */ -32768.0,
    /* invalidValue       */ -32768.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Geodetic (Geographic) Latitude
const PointAttributeInfo RDB_RIEGL_POF_LATITUDE = {
    /* name               */ "riegl.pof_latitude",
    /* title              */ "Latitude",
    /* tags               */ "",
    /* description        */ "Geodetic (Geographic) Latitude",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-9,
    /* minimumValue       */ -90.0,
    /* maximumValue       */ 90.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_POF_LATITUDE_NORTH = 90.0;
const double RDB_RIEGL_POF_LATITUDE_SOUTH = -90.0;

// Geodetic (Geographic) Longitude
const PointAttributeInfo RDB_RIEGL_POF_LONGITUDE = {
    /* name               */ "riegl.pof_longitude",
    /* title              */ "Longitude",
    /* tags               */ "",
    /* description        */ "Geodetic (Geographic) Longitude",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-9,
    /* minimumValue       */ -180.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_POF_LONGITUDE_EAST = 180.0;
const double RDB_RIEGL_POF_LONGITUDE_WEST = -180.0;

// Height (vertical distance) wrt. to the ellipsoid of the defined geodetic datum
const PointAttributeInfo RDB_RIEGL_POF_HEIGHT = {
    /* name               */ "riegl.pof_height",
    /* title              */ "Ellipsoidal Height",
    /* tags               */ "",
    /* description        */ "Height (vertical distance) wrt. to the ellipsoid of the defined geodetic datum",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 250.0e-6,
    /* minimumValue       */ -10000.0,
    /* maximumValue       */ 40000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Height of the geoid relative to the ellipsoid
const PointAttributeInfo RDB_RIEGL_POF_UNDULATION = {
    /* name               */ "riegl.pof_undulation",
    /* title              */ "Geoid Undulation",
    /* tags               */ "",
    /* description        */ "Height of the geoid relative to the ellipsoid",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -300.0,
    /* maximumValue       */ 300.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation about the body axis pointing in forward direction (x-axis for NED systems)
const PointAttributeInfo RDB_RIEGL_POF_ROLL = {
    /* name               */ "riegl.pof_roll",
    /* title              */ "Roll Angle",
    /* tags               */ "",
    /* description        */ "Rotation about the body axis pointing in forward direction (x-axis for NED systems)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -180.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation about the body axis pointing in transverse direction (y-axis for NED systems)
const PointAttributeInfo RDB_RIEGL_POF_PITCH = {
    /* name               */ "riegl.pof_pitch",
    /* title              */ "Pitch Angle",
    /* tags               */ "",
    /* description        */ "Rotation about the body axis pointing in transverse direction (y-axis for NED systems)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -180.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation about the body axis pointing in vertical direction (z-axis for NED systems)
const PointAttributeInfo RDB_RIEGL_POF_YAW = {
    /* name               */ "riegl.pof_yaw",
    /* title              */ "Yaw Angle",
    /* tags               */ "",
    /* description        */ "Rotation about the body axis pointing in vertical direction (z-axis for NED systems)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 2D angle of the course over ground (0 deg: north, 90 deg: east)
const PointAttributeInfo RDB_RIEGL_POF_COURSE = {
    /* name               */ "riegl.pof_course",
    /* title              */ "Course Angle",
    /* tags               */ "",
    /* description        */ "2D angle of the course over ground (0 deg: north, 90 deg: east)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Position of the platform in the coordinate system defined by the geo_tag (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_POF_XYZ = {
    /* name               */ "riegl.pof_xyz",
    /* title              */ "XYZ",
    /* tags               */ "position, transform",
    /* description        */ "Position of the platform in the coordinate system defined by the geo_tag (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -535000.0,
    /* maximumValue       */ 535000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_POF_XYZ_X = 0;
const size_t RDB_RIEGL_POF_XYZ_Y = 1;
const size_t RDB_RIEGL_POF_XYZ_Z = 2;

// Rotation about y-axis of the coordinate system defined by the geo_tag
const PointAttributeInfo RDB_RIEGL_POF_ROLL_NED = {
    /* name               */ "riegl.pof_roll_ned",
    /* title              */ "Roll Angle NED",
    /* tags               */ "",
    /* description        */ "Rotation about y-axis of the coordinate system defined by the geo_tag",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -180.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation about x-axis of the coordinate system defined by the geo_tag
const PointAttributeInfo RDB_RIEGL_POF_PITCH_NED = {
    /* name               */ "riegl.pof_pitch_ned",
    /* title              */ "Pitch Angle NED",
    /* tags               */ "",
    /* description        */ "Rotation about x-axis of the coordinate system defined by the geo_tag",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -180.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation about inverted z-axis of the coordinate system defined by the geo_tag
const PointAttributeInfo RDB_RIEGL_POF_YAW_NED = {
    /* name               */ "riegl.pof_yaw_ned",
    /* title              */ "Yaw Angle NED",
    /* tags               */ "",
    /* description        */ "Rotation about inverted z-axis of the coordinate system defined by the geo_tag",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Time-smoothed velocity vector of the platform in the body coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_POF_XYZ_VELOCITY = {
    /* name               */ "riegl.pof_xyz_velocity",
    /* title              */ "XYZ Velocity",
    /* tags               */ "direction, transform",
    /* description        */ "Time-smoothed velocity vector of the platform in the body coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m/s",
    /* length             */ 3,
    /* resolution         */ 0.01,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Time-smoothed velocity vector of the platform in the local NED system (0: N, 1: E, 2: D)
const PointAttributeInfo RDB_RIEGL_POF_XYZ_NED_VELOCITY = {
    /* name               */ "riegl.pof_xyz_ned_velocity",
    /* title              */ "XYZ NED Velocity",
    /* tags               */ "direction",
    /* description        */ "Time-smoothed velocity vector of the platform in the local NED system (0: N, 1: E, 2: D)",
    /* unitSymbol         */ "m/s",
    /* length             */ 3,
    /* resolution         */ 0.01,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Time-smoothed velocity vector of the platform wrt. levelled vehicle coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_POF_XYZ_LVCS_VELOCITY = {
    /* name               */ "riegl.pof_xyz_lvcs_velocity",
    /* title              */ "XYZ LVCS Velocity",
    /* tags               */ "direction",
    /* description        */ "Time-smoothed velocity vector of the platform wrt. levelled vehicle coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m/s",
    /* length             */ 3,
    /* resolution         */ 0.01,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation velocity about x-axis of the body coordinate system
const PointAttributeInfo RDB_RIEGL_POF_ROLL_VELOCITY = {
    /* name               */ "riegl.pof_roll_velocity",
    /* title              */ "Roll Angle Velocity",
    /* tags               */ "",
    /* description        */ "Rotation velocity about x-axis of the body coordinate system",
    /* unitSymbol         */ "deg/s",
    /* length             */ 1,
    /* resolution         */ 0.05,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation velocity about y-axis of the body coordinate system
const PointAttributeInfo RDB_RIEGL_POF_PITCH_VELOCITY = {
    /* name               */ "riegl.pof_pitch_velocity",
    /* title              */ "Pitch Angle Velocity",
    /* tags               */ "",
    /* description        */ "Rotation velocity about y-axis of the body coordinate system",
    /* unitSymbol         */ "deg/s",
    /* length             */ 1,
    /* resolution         */ 0.05,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Rotation velocity about z-axis of the body coordinate system
const PointAttributeInfo RDB_RIEGL_POF_YAW_VELOCITY = {
    /* name               */ "riegl.pof_yaw_velocity",
    /* title              */ "Yaw Angle Velocity",
    /* tags               */ "",
    /* description        */ "Rotation velocity about z-axis of the body coordinate system",
    /* unitSymbol         */ "deg/s",
    /* length             */ 1,
    /* resolution         */ 0.05,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Velocity vector in the local NED system (0: N, 1: E, 2: D)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_NED = {
    /* name               */ "riegl.pof_velocity_ned",
    /* title              */ "Velocity NED",
    /* tags               */ "direction",
    /* description        */ "Velocity vector in the local NED system (0: N, 1: E, 2: D)",
    /* unitSymbol         */ "m/s",
    /* length             */ 3,
    /* resolution         */ 0.001,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Velocity in North direction (standard deviation)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH = {
    /* name               */ "riegl.pof_velocity_accuracy_north",
    /* title              */ "Velocity Accuracy North",
    /* tags               */ "",
    /* description        */ "Accuracy of Velocity in North direction (standard deviation)",
    /* unitSymbol         */ "m/s",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Velocity in East direction (standard deviation)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST = {
    /* name               */ "riegl.pof_velocity_accuracy_east",
    /* title              */ "Velocity Accuracy East",
    /* tags               */ "",
    /* description        */ "Accuracy of Velocity in East direction (standard deviation)",
    /* unitSymbol         */ "m/s",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Velocity in Down direction (standard deviation)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN = {
    /* name               */ "riegl.pof_velocity_accuracy_down",
    /* title              */ "Velocity Accuracy Down",
    /* tags               */ "",
    /* description        */ "Accuracy of Velocity in Down direction (standard deviation)",
    /* unitSymbol         */ "m/s",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Velocity in North/East direction (signed square root of covariance)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST = {
    /* name               */ "riegl.pof_velocity_accuracy_north_east",
    /* title              */ "Velocity Accuracy North/East",
    /* tags               */ "",
    /* description        */ "Accuracy of Velocity in North/East direction (signed square root of covariance)",
    /* unitSymbol         */ "m/s",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Velocity in East/Down direction (signed square root of covariance)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN = {
    /* name               */ "riegl.pof_velocity_accuracy_east_down",
    /* title              */ "Velocity Accuracy East/Down",
    /* tags               */ "",
    /* description        */ "Accuracy of Velocity in East/Down direction (signed square root of covariance)",
    /* unitSymbol         */ "m/s",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Velocity in Down/North direction (signed square root of covariance)
const PointAttributeInfo RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH = {
    /* name               */ "riegl.pof_velocity_accuracy_down_north",
    /* title              */ "Velocity Accuracy Down/North",
    /* tags               */ "",
    /* description        */ "Accuracy of Velocity in Down/North direction (signed square root of covariance)",
    /* unitSymbol         */ "m/s",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -200.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Water surface intersection point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_HYDRO_INTERSECTION_POINT = {
    /* name               */ "riegl.hydro_intersection_point",
    /* title              */ "Water Surface Intersection Point",
    /* tags               */ "position, transform",
    /* description        */ "Water surface intersection point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -535000.0,
    /* maximumValue       */ 535000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_HYDRO_INTERSECTION_POINT_X = 0;
const size_t RDB_RIEGL_HYDRO_INTERSECTION_POINT_Y = 1;
const size_t RDB_RIEGL_HYDRO_INTERSECTION_POINT_Z = 2;

// Water surface intersection normal vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_HYDRO_INTERSECTION_NORMAL = {
    /* name               */ "riegl.hydro_intersection_normal",
    /* title              */ "Water Surface Intersection Normal",
    /* tags               */ "direction, transform",
    /* description        */ "Water surface intersection normal vector wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_HYDRO_INTERSECTION_NORMAL_X = 0;
const size_t RDB_RIEGL_HYDRO_INTERSECTION_NORMAL_Y = 1;
const size_t RDB_RIEGL_HYDRO_INTERSECTION_NORMAL_Z = 2;

// Water surface model uncertainty (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_HYDRO_WSM_UNCERTAINTY = {
    /* name               */ "riegl.hydro_wsm_uncertainty",
    /* title              */ "Water Surface Model Uncertainty",
    /* tags               */ "direction, transform",
    /* description        */ "Water surface model uncertainty (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.0001,
    /* minimumValue       */ -3.2768,
    /* maximumValue       */ 3.2767,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_HYDRO_WSM_UNCERTAINTY_X = 0;
const size_t RDB_RIEGL_HYDRO_WSM_UNCERTAINTY_Y = 1;
const size_t RDB_RIEGL_HYDRO_WSM_UNCERTAINTY_Z = 2;

// Accuracy of Cartesian coordinates (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_XYZ_ACCURACIES = {
    /* name               */ "riegl.xyz_accuracies",
    /* title              */ "XYZ Accuracies",
    /* tags               */ "direction, transform",
    /* description        */ "Accuracy of Cartesian coordinates (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Direction of zenith vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const PointAttributeInfo RDB_RIEGL_ZENITH_VECTOR = {
    /* name               */ "riegl.zenith_vector",
    /* title              */ "Zenith Vector",
    /* tags               */ "direction, transform",
    /* description        */ "Direction of zenith vector wrt. application coordinate system (0: X, 1: Y, 2: Z)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 3.125e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Time"
//______________________________________________________________________________
//

// Laser shot timestamp (in units of 4 pico seconds = 4e-12 s)
const PointAttributeInfo RDB_RIEGL_SHOT_TIMESTAMP_HR = {
    /* name               */ "riegl.shot_timestamp_hr",
    /* title              */ "Laser Shot Timestamp",
    /* tags               */ "timestamp",
    /* description        */ "Laser shot timestamp (in units of 4 pico seconds = 4e-12 s)",
    /* unitSymbol         */ "4ps",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e18,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_SHOT_TIMESTAMP_HR_RESOLUTION = 4.0e-12;

// Laser shot timestamp (100 nano seconds resolution)
const PointAttributeInfo RDB_RIEGL_TIMESTAMP = {
    /* name               */ "riegl.timestamp",
    /* title              */ "Timestamp",
    /* tags               */ "timestamp",
    /* description        */ "Laser shot timestamp (100 nano seconds resolution)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 1.0e-7,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e8,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Start of waveform sample block relative to laser shot timestamp (in units of 4 pico seconds = 4e-12 s)
const PointAttributeInfo RDB_RIEGL_WFM_SBL_TIME_OFFSET = {
    /* name               */ "riegl.wfm_sbl_time_offset",
    /* title              */ "Waveform Sample Block Time Offset",
    /* tags               */ "",
    /* description        */ "Start of waveform sample block relative to laser shot timestamp (in units of 4 pico seconds = 4e-12 s)",
    /* unitSymbol         */ "4ps",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -2.0e9,
    /* maximumValue       */ 2.0e9,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_WFM_SBL_TIME_OFFSET_RESOLUTION = 4.0e-12;

// Position of echo relative to start of waveform sample block (in units of 4 pico seconds = 4e-12 s)
const PointAttributeInfo RDB_RIEGL_WFM_ECHO_TIME_OFFSET = {
    /* name               */ "riegl.wfm_echo_time_offset",
    /* title              */ "Echo Time Offset",
    /* tags               */ "",
    /* description        */ "Position of echo relative to start of waveform sample block (in units of 4 pico seconds = 4e-12 s)",
    /* unitSymbol         */ "4ps",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -2.0e9,
    /* maximumValue       */ 2.0e9,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_WFM_ECHO_TIME_OFFSET_RESOLUTION = 4.0e-12;

// External timestamp (100 nano seconds resolution)
const PointAttributeInfo RDB_RIEGL_PPS_TIMESTAMP_EXTERN = {
    /* name               */ "riegl.pps_timestamp_extern",
    /* title              */ "External Timestamp",
    /* tags               */ "timestamp",
    /* description        */ "External timestamp (100 nano seconds resolution)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 1.0e-7,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e8,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Internal timestamp (in units of 4 pico seconds = 4e-12 s)
const PointAttributeInfo RDB_RIEGL_PPS_TIMESTAMP_INTERN = {
    /* name               */ "riegl.pps_timestamp_intern",
    /* title              */ "Internal Timestamp",
    /* tags               */ "",
    /* description        */ "Internal timestamp (in units of 4 pico seconds = 4e-12 s)",
    /* unitSymbol         */ "4ps",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e18,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_PPS_TIMESTAMP_INTERN_RESOLUTION = 4.0e-12;

// Timestamp of event (seconds since epoch defined in riegl.time_base)
const PointAttributeInfo RDB_RIEGL_EVENT_TIMESTAMP = {
    /* name               */ "riegl.event_timestamp",
    /* title              */ "Event Timestamp",
    /* tags               */ "timestamp",
    /* description        */ "Timestamp of event (seconds since epoch defined in riegl.time_base)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 1e-07,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 900000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Minimum laser shot timestamp within voxel (100 nano seconds resolution)
const PointAttributeInfo RDB_RIEGL_TIMESTAMP_MIN = {
    /* name               */ "riegl.timestamp_min",
    /* title              */ "Timestamp Minimum",
    /* tags               */ "timestamp",
    /* description        */ "Minimum laser shot timestamp within voxel (100 nano seconds resolution)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 1.0e-7,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e8,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Maximum laser shot timestamp within voxel (100 nano seconds resolution)
const PointAttributeInfo RDB_RIEGL_TIMESTAMP_MAX = {
    /* name               */ "riegl.timestamp_max",
    /* title              */ "Timestamp Maximum",
    /* tags               */ "timestamp",
    /* description        */ "Maximum laser shot timestamp within voxel (100 nano seconds resolution)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 1.0e-7,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e8,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Timestamp (100 nano seconds resolution)
const PointAttributeInfo RDB_RIEGL_POF_TIMESTAMP = {
    /* name               */ "riegl.pof_timestamp",
    /* title              */ "Timestamp",
    /* tags               */ "timestamp",
    /* description        */ "Timestamp (100 nano seconds resolution)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 1.0e-7,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 9.0e8,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Date of acquisition (0: year, 1: month [1-12], 2: day [1-31], not specified if any is 0)
const PointAttributeInfo RDB_RIEGL_ACQUISITION_DATE = {
    /* name               */ "riegl.acquisition_date",
    /* title              */ "Acquisition Date",
    /* tags               */ "",
    /* description        */ "Date of acquisition (0: year, 1: month [1-12], 2: day [1-31], not specified if any is 0)",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4095.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_ACQUISITION_DATE_YEAR = 0;
const size_t RDB_RIEGL_ACQUISITION_DATE_MONTH = 1;
const size_t RDB_RIEGL_ACQUISITION_DATE_DAY = 2;

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Primary Attributes"
//______________________________________________________________________________
//

// Target surface reflectance
const PointAttributeInfo RDB_RIEGL_REFLECTANCE = {
    /* name               */ "riegl.reflectance",
    /* title              */ "Reflectance",
    /* tags               */ "",
    /* description        */ "Target surface reflectance",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -327.68,
    /* maximumValue       */ 327.67,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -327.68,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Echo signal amplitude
const PointAttributeInfo RDB_RIEGL_AMPLITUDE = {
    /* name               */ "riegl.amplitude",
    /* title              */ "Amplitude",
    /* tags               */ "",
    /* description        */ "Echo signal amplitude",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -327.68,
    /* maximumValue       */ 327.67,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -327.68,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Uncalibrated echo signal amplitude (for legacy RIEGL LMS instruments)
const PointAttributeInfo RDB_RIEGL_INTENSITY = {
    /* name               */ "riegl.intensity",
    /* title              */ "Intensity",
    /* tags               */ "",
    /* description        */ "Uncalibrated echo signal amplitude (for legacy RIEGL LMS instruments)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Gain of photodiode
const PointAttributeInfo RDB_RIEGL_GAIN = {
    /* name               */ "riegl.gain",
    /* title              */ "Gain",
    /* tags               */ "",
    /* description        */ "Gain of photodiode",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -327.68,
    /* maximumValue       */ 327.67,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Pulse shape deviation (negative means unavailable)
const PointAttributeInfo RDB_RIEGL_DEVIATION = {
    /* name               */ "riegl.deviation",
    /* title              */ "Deviation",
    /* tags               */ "",
    /* description        */ "Pulse shape deviation (negative means unavailable)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 32767.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const int16_t RDB_RIEGL_DEVIATION_UNAVAILABLE = -1;

// Pulse width (full width at half maximum, for Q-Series instruments, 0 = invalid)
const PointAttributeInfo RDB_RIEGL_PULSE_WIDTH = {
    /* name               */ "riegl.pulse_width",
    /* title              */ "Pulse Width",
    /* tags               */ "",
    /* description        */ "Pulse width (full width at half maximum, for Q-Series instruments, 0 = invalid)",
    /* unitSymbol         */ "ns",
    /* length             */ 1,
    /* resolution         */ 0.1,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 6553.5,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_PULSE_WIDTH_INVALID = 0.0;

// Point class number
const PointAttributeInfo RDB_RIEGL_CLASS = {
    /* name               */ "riegl.class",
    /* title              */ "Point Class",
    /* tags               */ "enumeration",
    /* description        */ "Point class number",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=Created, never classified\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Point color derived from digital camera, 0: Red, 1: Green, 2: Blue, 3: Alpha (0 = no color); additional colors (if any) are stored in riegl.rgba_2, riegl.rgba_3, ...
const PointAttributeInfo RDB_RIEGL_RGBA = {
    /* name               */ "riegl.rgba",
    /* title              */ "True Color",
    /* tags               */ "color",
    /* description        */ "Point color derived from digital camera, 0: Red, 1: Green, 2: Blue, 3: Alpha (0 = no color); additional colors (if any) are stored in riegl.rgba_2, riegl.rgba_3, ...",
    /* unitSymbol         */ "",
    /* length             */ 4,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_RGBA_RED = 0;
const size_t RDB_RIEGL_RGBA_GREEN = 1;
const size_t RDB_RIEGL_RGBA_BLUE = 2;
const size_t RDB_RIEGL_RGBA_ALPHA = 3;

// Point near infrared (NIR) value derived from NIR camera
const PointAttributeInfo RDB_RIEGL_NIR = {
    /* name               */ "riegl.nir",
    /* title              */ "Near Infrared Brightness",
    /* tags               */ "",
    /* description        */ "Point near infrared (NIR) value derived from NIR camera",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Target temperature measured by thermal camera
const PointAttributeInfo RDB_RIEGL_TEMPERATURE = {
    /* name               */ "riegl.temperature",
    /* title              */ "Temperature",
    /* tags               */ "",
    /* description        */ "Target temperature measured by thermal camera",
    /* unitSymbol         */ "°C",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -273.15,
    /* maximumValue       */ 4.0e6,
    /* defaultValue       */ 4.0e6,
    /* invalidValue       */ 4.0e6,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// DMI counter value (unwrapped)
const PointAttributeInfo RDB_RIEGL_DMI_COUNT = {
    /* name               */ "riegl.dmi_count",
    /* title              */ "DMI Counter Value",
    /* tags               */ "",
    /* description        */ "DMI counter value (unwrapped)",
    /* unitSymbol         */ "1",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -100000000000.0,
    /* maximumValue       */ 100000000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// DMI path length (uncalibrated)
const PointAttributeInfo RDB_RIEGL_DMI_PATH_LENGTH_RAW = {
    /* name               */ "riegl.dmi_path_length_raw",
    /* title              */ "DMI Path Length",
    /* tags               */ "",
    /* description        */ "DMI path length (uncalibrated)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -10000000.0,
    /* maximumValue       */ 10000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated deviation of scale of DMI from 1 with respect to the riegl.dmi_path_length_raw attribute in the dmix file
const PointAttributeInfo RDB_RIEGL_DMI_SCALE = {
    /* name               */ "riegl.dmi_scale",
    /* title              */ "DMI Scale",
    /* tags               */ "",
    /* description        */ "Estimated deviation of scale of DMI from 1 with respect to the riegl.dmi_path_length_raw attribute in the dmix file",
    /* unitSymbol         */ "1",
    /* length             */ 1,
    /* resolution         */ 1e-05,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Index of assigned MTA zone. Nearest MTA zone has index 1
const PointAttributeInfo RDB_RIEGL_MTA_ZONE = {
    /* name               */ "riegl.mta_zone",
    /* title              */ "MTA Zone Assigned",
    /* tags               */ "enumeration",
    /* description        */ "Index of assigned MTA zone. Nearest MTA zone has index 1",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 1.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 1 for points with uncertain MTA zone assignment, 0 for certainly assigned MTA zone
const PointAttributeInfo RDB_RIEGL_MTA_UNCERTAIN_POINT = {
    /* name               */ "riegl.mta_uncertain_point",
    /* title              */ "MTA Uncertain Point",
    /* tags               */ "boolean",
    /* description        */ "1 for points with uncertain MTA zone assignment, 0 for certainly assigned MTA zone",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_MTA_UNCERTAIN_POINT_FALSE = 0;
const uint8_t RDB_RIEGL_MTA_UNCERTAIN_POINT_TRUE = 1;

// 1 for all points originating from a full waveform analysis (FWA), 0 for all points originating from online waveform processing (OWP)
const PointAttributeInfo RDB_RIEGL_FWA = {
    /* name               */ "riegl.fwa",
    /* title              */ "Full Waveform Analysis",
    /* tags               */ "boolean",
    /* description        */ "1 for all points originating from a full waveform analysis (FWA), 0 for all points originating from online waveform processing (OWP)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=Origin OWP\n1=Origin FWA\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_FWA_ONLINE_WAVEFORM_PROCESSING = 0;
const uint8_t RDB_RIEGL_FWA_FULL_WAVEFORM_ANALYSIS = 1;

// Background Radiation (for VZ-400-HT/HAT only)
const PointAttributeInfo RDB_RIEGL_BACKGROUND_RADIATION = {
    /* name               */ "riegl.background_radiation",
    /* title              */ "Background Radiation",
    /* tags               */ "",
    /* description        */ "Background Radiation (for VZ-400-HT/HAT only)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Background Radiation (for VZ-400-HTo only, Si-PD)
const PointAttributeInfo RDB_RIEGL_BACKGROUND_RADIATION_SI = {
    /* name               */ "riegl.background_radiation_si",
    /* title              */ "Background Radiation Si-PD",
    /* tags               */ "",
    /* description        */ "Background Radiation (for VZ-400-HTo only, Si-PD)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 65535.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Background Radiation (for VZ-400-HTo only, InGaAs-PD)
const PointAttributeInfo RDB_RIEGL_BACKGROUND_RADIATION_INGAAS = {
    /* name               */ "riegl.background_radiation_ingaas",
    /* title              */ "Background Radiation InGaAs-PD",
    /* tags               */ "",
    /* description        */ "Background Radiation (for VZ-400-HTo only, InGaAs-PD)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 65535.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated temperature derived from Background Radiation Si-PD
const PointAttributeInfo RDB_RIEGL_TEMPERATURE_ESTIMATED_SI = {
    /* name               */ "riegl.temperature_estimated_si",
    /* title              */ "Temperature Estimated Si-PD",
    /* tags               */ "",
    /* description        */ "Estimated temperature derived from Background Radiation Si-PD",
    /* unitSymbol         */ "°C",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated temperature derived from Background Radiation InGaAs-PD
const PointAttributeInfo RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS = {
    /* name               */ "riegl.temperature_estimated_ingaas",
    /* title              */ "Temperature Estimated InGaAs-PD",
    /* tags               */ "",
    /* description        */ "Estimated temperature derived from Background Radiation InGaAs-PD",
    /* unitSymbol         */ "°C",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Estimated temperature derived from difference in Background Radiation InGaAs and Si
const PointAttributeInfo RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI = {
    /* name               */ "riegl.temperature_estimated_ingaas_si",
    /* title              */ "Temperature Estimated InGaAs/Si-PD",
    /* tags               */ "",
    /* description        */ "Estimated temperature derived from difference in Background Radiation InGaAs and Si",
    /* unitSymbol         */ "°C",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// nth target of a laser-shot (0 = unknown, 1 = first target, ...)
const PointAttributeInfo RDB_RIEGL_TARGET_INDEX = {
    /* name               */ "riegl.target_index",
    /* title              */ "Target Index",
    /* tags               */ "",
    /* description        */ "nth target of a laser-shot (0 = unknown, 1 = first target, ...)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of targets of a laser-shot (0 = unknown, 1 = one target, ...)
const PointAttributeInfo RDB_RIEGL_TARGET_COUNT = {
    /* name               */ "riegl.target_count",
    /* title              */ "Target Count",
    /* tags               */ "",
    /* description        */ "Number of targets of a laser-shot (0 = unknown, 1 = one target, ...)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Target rank among all targets of a laser-shot (0 = unknown, 1 = single, 2 = first, 3 = intermediate, 4 = last target)
const PointAttributeInfo RDB_RIEGL_TARGET_TYPE = {
    /* name               */ "riegl.target_type",
    /* title              */ "Target Type",
    /* tags               */ "enumeration",
    /* description        */ "Target rank among all targets of a laser-shot (0 = unknown, 1 = single, 2 = first, 3 = intermediate, 4 = last target)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "1=Single target\n2=First target\n3=Intermediate target\n4=Last target\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_TARGET_TYPE_UNKNOWN = 0;
const uint8_t RDB_RIEGL_TARGET_TYPE_SINGLE = 1;
const uint8_t RDB_RIEGL_TARGET_TYPE_FIRST = 2;
const uint8_t RDB_RIEGL_TARGET_TYPE_INTERMEDIATE = 3;
const uint8_t RDB_RIEGL_TARGET_TYPE_LAST = 4;

// Identifier of first echo that belongs to the laser shot (0 = invalid). This is not an array index but the value of riegl.id of the echo.
const PointAttributeInfo RDB_RIEGL_ECHO_FIRST = {
    /* name               */ "riegl.echo_first",
    /* title              */ "Echo First",
    /* tags               */ "",
    /* description        */ "Identifier of first echo that belongs to the laser shot (0 = invalid). This is not an array index but the value of riegl.id of the echo.",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_ECHO_FIRST_INVALID = 0;

// Number of echoes that belong to the laser shot
const PointAttributeInfo RDB_RIEGL_ECHO_COUNT = {
    /* name               */ "riegl.echo_count",
    /* title              */ "Echo Count",
    /* tags               */ "",
    /* description        */ "Number of echoes that belong to the laser shot",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Height at center of grid cell
const PointAttributeInfo RDB_RIEGL_HEIGHT_CENTER = {
    /* name               */ "riegl.height_center",
    /* title              */ "Height Center",
    /* tags               */ "",
    /* description        */ "Height at center of grid cell",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -100000.0,
    /* maximumValue       */ 100000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Mean height within grid cell
const PointAttributeInfo RDB_RIEGL_HEIGHT_MEAN = {
    /* name               */ "riegl.height_mean",
    /* title              */ "Height Mean",
    /* tags               */ "",
    /* description        */ "Mean height within grid cell",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -100000.0,
    /* maximumValue       */ 100000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Minimum height within grid cell
const PointAttributeInfo RDB_RIEGL_HEIGHT_MIN = {
    /* name               */ "riegl.height_min",
    /* title              */ "Height Minimum",
    /* tags               */ "",
    /* description        */ "Minimum height within grid cell",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -100000.0,
    /* maximumValue       */ 100000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Maximum height within grid cell
const PointAttributeInfo RDB_RIEGL_HEIGHT_MAX = {
    /* name               */ "riegl.height_max",
    /* title              */ "Height Maximum",
    /* tags               */ "",
    /* description        */ "Maximum height within grid cell",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -100000.0,
    /* maximumValue       */ 100000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of points this point represents (e.g. points combined to voxels or plane patches, 0 = unknown)
const PointAttributeInfo RDB_RIEGL_POINT_COUNT = {
    /* name               */ "riegl.point_count",
    /* title              */ "Point Count",
    /* tags               */ "",
    /* description        */ "Number of points this point represents (e.g. points combined to voxels or plane patches, 0 = unknown)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4294967295.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint32_t RDB_RIEGL_POINT_COUNT_UNKNOWN = 0;

// Number of total points in a grid cell this point represents.
const PointAttributeInfo RDB_RIEGL_POINT_COUNT_GRID_CELL = {
    /* name               */ "riegl.point_count_grid_cell",
    /* title              */ "Point Count of Grid Cell",
    /* tags               */ "",
    /* description        */ "Number of total points in a grid cell this point represents.",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4294967295.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint32_t RDB_RIEGL_POINT_COUNT_GRID_CELL_UNKNOWN = 0;

// Volume extents along 0: riegl.pca_axis_max, 1: riegl.pca_axis_min x riegl.pca_axis_max, 2: riegl.pca_axis_min
const PointAttributeInfo RDB_RIEGL_PCA_EXTENTS = {
    /* name               */ "riegl.pca_extents",
    /* title              */ "PCA Extents",
    /* tags               */ "",
    /* description        */ "Volume extents along 0: riegl.pca_axis_max, 1: riegl.pca_axis_min x riegl.pca_axis_max, 2: riegl.pca_axis_min",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Representative volume of the neighborhood, calculated as product of the components of riegl.pca_extents
const PointAttributeInfo RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL = {
    /* name               */ "riegl.mta_uncertain_point_vol",
    /* title              */ "MTA Uncertain Point Volume",
    /* tags               */ "",
    /* description        */ "Representative volume of the neighborhood, calculated as product of the components of riegl.pca_extents",
    /* unitSymbol         */ "dBm³",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -120.0,
    /* maximumValue       */ 120.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -120.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Difference in representative volume of the neighborhood to next smallest volume of point from the same echo but different MTA-zone
const PointAttributeInfo RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF = {
    /* name               */ "riegl.mta_uncertain_point_vol_diff",
    /* title              */ "Difference in MTA Uncertain Point Volume",
    /* tags               */ "",
    /* description        */ "Difference in representative volume of the neighborhood to next smallest volume of point from the same echo but different MTA-zone",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 240.0,
    /* defaultValue       */ 240.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Volume extents along riegl.pca_axis_min or riegl.surface_normal (result of PCA)
const PointAttributeInfo RDB_RIEGL_PCA_THICKNESS = {
    /* name               */ "riegl.pca_thickness",
    /* title              */ "PCA Thickness",
    /* tags               */ "",
    /* description        */ "Volume extents along riegl.pca_axis_min or riegl.surface_normal (result of PCA)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Representative diameter of the neighborhood, calculated as sqrt(12*sum_of_eigenvalues/number_of_points)
const PointAttributeInfo RDB_RIEGL_PCA_DIAMETER = {
    /* name               */ "riegl.pca_diameter",
    /* title              */ "PCA Diameter of Neighborhood",
    /* tags               */ "",
    /* description        */ "Representative diameter of the neighborhood, calculated as sqrt(12*sum_of_eigenvalues/number_of_points)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// The smallest eigenvalue from PCA normalized by the sum of all eigenvalues
const PointAttributeInfo RDB_RIEGL_EIGENVALUE_NORM_MIN = {
    /* name               */ "riegl.eigenvalue_norm_min",
    /* title              */ "PCA Minimum Eigen Value Normalized",
    /* tags               */ "",
    /* description        */ "The smallest eigenvalue from PCA normalized by the sum of all eigenvalues",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.52591e-05,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// The largest eigenvalue from PCA normalized by the sum of all eigenvalues
const PointAttributeInfo RDB_RIEGL_EIGENVALUE_NORM_MAX = {
    /* name               */ "riegl.eigenvalue_norm_max",
    /* title              */ "PCA Maximum Eigen Value Normalized",
    /* tags               */ "",
    /* description        */ "The largest eigenvalue from PCA normalized by the sum of all eigenvalues",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.52591e-05,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Standard deviation, e.g. of residual point-to-plane distances (0 = unknown)
const PointAttributeInfo RDB_RIEGL_STD_DEV = {
    /* name               */ "riegl.std_dev",
    /* title              */ "Standard Deviation",
    /* tags               */ "",
    /* description        */ "Standard deviation, e.g. of residual point-to-plane distances (0 = unknown)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.0001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65.535,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_STD_DEV_UNKNOWN = 0.0;

// Confidence ellipse for the normal vector of a plane patch (0: up-axis, 1: width-axis, tilt angle wrt. up-axis)
const PointAttributeInfo RDB_RIEGL_PLANE_CONFIDENCE_NORMAL = {
    /* name               */ "riegl.plane_confidence_normal",
    /* title              */ "Confidence Normal Direction",
    /* tags               */ "",
    /* description        */ "Confidence ellipse for the normal vector of a plane patch (0: up-axis, 1: width-axis, tilt angle wrt. up-axis)",
    /* unitSymbol         */ "deg",
    /* length             */ 3,
    /* resolution         */ 0.006,
    /* minimumValue       */ -180.006,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ -180.006,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Plane patch slope class number. Meaning see metadata riegl.plane_slope_class_info (0 = invalid)
const PointAttributeInfo RDB_RIEGL_PLANE_SLOPE_CLASS = {
    /* name               */ "riegl.plane_slope_class",
    /* title              */ "Plane Patch Slope Class",
    /* tags               */ "enumeration",
    /* description        */ "Plane patch slope class number. Meaning see metadata riegl.plane_slope_class_info (0 = invalid)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 8x8 occupancy matrix indicating point support of plane patch
const PointAttributeInfo RDB_RIEGL_PLANE_OCCUPANCY = {
    /* name               */ "riegl.plane_occupancy",
    /* title              */ "Plane Patch Occupancy",
    /* tags               */ "",
    /* description        */ "8x8 occupancy matrix indicating point support of plane patch",
    /* unitSymbol         */ "",
    /* length             */ 8,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Size of plane patch along the longer edge (0 = unknown)
const PointAttributeInfo RDB_RIEGL_PLANE_WIDTH = {
    /* name               */ "riegl.plane_width",
    /* title              */ "Plane Patch Width",
    /* tags               */ "",
    /* description        */ "Size of plane patch along the longer edge (0 = unknown)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_PLANE_WIDTH_UNKNOWN = 0.0;

// Size of plane patch along the shorter edge (0 = unknown)
const PointAttributeInfo RDB_RIEGL_PLANE_HEIGHT = {
    /* name               */ "riegl.plane_height",
    /* title              */ "Plane Patch Height",
    /* tags               */ "",
    /* description        */ "Size of plane patch along the shorter edge (0 = unknown)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_PLANE_HEIGHT_UNKNOWN = 0.0;

// For merged plane patches, the number of plane patches the merged plane patch is based on
const PointAttributeInfo RDB_RIEGL_PLANE_COUNT = {
    /* name               */ "riegl.plane_count",
    /* title              */ "Plane Patch Count",
    /* tags               */ "",
    /* description        */ "For merged plane patches, the number of plane patches the merged plane patch is based on",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// The number of source plane patch files the plane patch has matches to
const PointAttributeInfo RDB_RIEGL_MATCH_COUNT = {
    /* name               */ "riegl.match_count",
    /* title              */ "Plane Patch Match Count",
    /* tags               */ "",
    /* description        */ "The number of source plane patch files the plane patch has matches to",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Distance between the origins of two plane patches, projected onto the average of their normal vectors
const PointAttributeInfo RDB_RIEGL_PLANE_PATCH_DISTANCE = {
    /* name               */ "riegl.plane_patch_distance",
    /* title              */ "Plane Patch Distance",
    /* tags               */ "",
    /* description        */ "Distance between the origins of two plane patches, projected onto the average of their normal vectors",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -2100.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ -2100.0,
    /* invalidValue       */ -2100.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_PLANE_PATCH_DISTANCE_LOWEST = -2000.0;

// Distance between the origins of two plane patches, projected onto the plane defined by the average of their normal vectors
const PointAttributeInfo RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE = {
    /* name               */ "riegl.plane_patch_lateral_distance",
    /* title              */ "Plane Patch Lateral Distance",
    /* tags               */ "",
    /* description        */ "Distance between the origins of two plane patches, projected onto the plane defined by the average of their normal vectors",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -2100.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ -2100.0,
    /* invalidValue       */ -2100.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE_LOWEST = -2000.0;

// Angle between the normal vectors of two plane patches
const PointAttributeInfo RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE = {
    /* name               */ "riegl.plane_patch_angular_distance",
    /* title              */ "Plane Patch Angular Distance",
    /* tags               */ "",
    /* description        */ "Angle between the normal vectors of two plane patches",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Radius of cylinder
const PointAttributeInfo RDB_RIEGL_CYLINDER_RADIUS = {
    /* name               */ "riegl.cylinder_radius",
    /* title              */ "Cylinder Radius",
    /* tags               */ "",
    /* description        */ "Radius of cylinder",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 100.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Height of cylinder (cylinder origin is at half height)
const PointAttributeInfo RDB_RIEGL_CYLINDER_HEIGHT = {
    /* name               */ "riegl.cylinder_height",
    /* title              */ "Cylinder Height",
    /* tags               */ "",
    /* description        */ "Height of cylinder (cylinder origin is at half height)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence of estimated cylinder origin (perpendicular to axis)
const PointAttributeInfo RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE = {
    /* name               */ "riegl.cylinder_origin_confidence",
    /* title              */ "Cylinder Origin Confidence",
    /* tags               */ "",
    /* description        */ "Confidence of estimated cylinder origin (perpendicular to axis)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.0001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 10.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence of estimated cylinder radius
const PointAttributeInfo RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE = {
    /* name               */ "riegl.cylinder_radius_confidence",
    /* title              */ "Cylinder Radius Confidence",
    /* tags               */ "",
    /* description        */ "Confidence of estimated cylinder radius",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.0001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 10.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence of estimated cylinder axis direction
const PointAttributeInfo RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE = {
    /* name               */ "riegl.cylinder_direction_confidence",
    /* title              */ "Cylinder Direction Confidence",
    /* tags               */ "",
    /* description        */ "Confidence of estimated cylinder axis direction",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.003,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of points used for cylinder fitting
const PointAttributeInfo RDB_RIEGL_CYLINDER_INLIER_COUNT = {
    /* name               */ "riegl.cylinder_inlier_count",
    /* title              */ "Cylinder Inlier Count",
    /* tags               */ "",
    /* description        */ "Number of points used for cylinder fitting",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 16000000,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of points not used for cylinder fitting
const PointAttributeInfo RDB_RIEGL_CYLINDER_OUTLIER_COUNT = {
    /* name               */ "riegl.cylinder_outlier_count",
    /* title              */ "Cylinder Outlier Count",
    /* tags               */ "",
    /* description        */ "Number of points not used for cylinder fitting",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 16000000,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy in North direction (standard deviation)
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_NORTH = {
    /* name               */ "riegl.pof_accuracy_north",
    /* title              */ "Accuracy North",
    /* tags               */ "",
    /* description        */ "Accuracy in North direction (standard deviation)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy in East direction (standard deviation)
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_EAST = {
    /* name               */ "riegl.pof_accuracy_east",
    /* title              */ "Accuracy East",
    /* tags               */ "",
    /* description        */ "Accuracy in East direction (standard deviation)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy in Down direction (standard deviation)
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_DOWN = {
    /* name               */ "riegl.pof_accuracy_down",
    /* title              */ "Accuracy Down",
    /* tags               */ "",
    /* description        */ "Accuracy in Down direction (standard deviation)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy in North/East direction (signed square root of covariance)
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_NORTH_EAST = {
    /* name               */ "riegl.pof_accuracy_north_east",
    /* title              */ "Accuracy North/East",
    /* tags               */ "",
    /* description        */ "Accuracy in North/East direction (signed square root of covariance)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy in East/Down direction (signed square root of covariance)
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_EAST_DOWN = {
    /* name               */ "riegl.pof_accuracy_east_down",
    /* title              */ "Accuracy East/Down",
    /* tags               */ "",
    /* description        */ "Accuracy in East/Down direction (signed square root of covariance)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy in Down/North direction (signed square root of covariance)
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_DOWN_NORTH = {
    /* name               */ "riegl.pof_accuracy_down_north",
    /* title              */ "Accuracy Down/North",
    /* tags               */ "",
    /* description        */ "Accuracy in Down/North direction (signed square root of covariance)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -1000.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Roll rotation
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_ROLL = {
    /* name               */ "riegl.pof_accuracy_roll",
    /* title              */ "Accuracy Roll",
    /* tags               */ "",
    /* description        */ "Accuracy of Roll rotation",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Pitch rotation
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_PITCH = {
    /* name               */ "riegl.pof_accuracy_pitch",
    /* title              */ "Accuracy Pitch",
    /* tags               */ "",
    /* description        */ "Accuracy of Pitch rotation",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Accuracy of Yaw rotation
const PointAttributeInfo RDB_RIEGL_POF_ACCURACY_YAW = {
    /* name               */ "riegl.pof_accuracy_yaw",
    /* title              */ "Accuracy Yaw",
    /* tags               */ "",
    /* description        */ "Accuracy of Yaw rotation",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Waveform sample block channel number (255 = invalid)
const PointAttributeInfo RDB_RIEGL_WFM_SBL_CHANNEL = {
    /* name               */ "riegl.wfm_sbl_channel",
    /* title              */ "Waveform Sample Block Channel",
    /* tags               */ "enumeration",
    /* description        */ "Waveform sample block channel number (255 = invalid)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 255.0,
    /* invalidValue       */ 255.0,
    /* namedValues        */ "0=High power\n1=Low power\n3=Reference pulse\n",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_WFM_SBL_CHANNEL_INVALID = 255;

// Waveform sample value mean
const PointAttributeInfo RDB_RIEGL_WFM_SBL_MEAN = {
    /* name               */ "riegl.wfm_sbl_mean",
    /* title              */ "Waveform Sample Value Mean",
    /* tags               */ "",
    /* description        */ "Waveform sample value mean",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 0.0625,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4095.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Waveform sample value standard deviation
const PointAttributeInfo RDB_RIEGL_WFM_SBL_STD_DEV = {
    /* name               */ "riegl.wfm_sbl_std_dev",
    /* title              */ "Waveform Sample Value Standard Deviation",
    /* tags               */ "",
    /* description        */ "Waveform sample value standard deviation",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 0.25,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Identifier of first waveform sample block that belongs to the laser shot (0 = invalid). This is not an array index but the value of riegl.id of the block.
const PointAttributeInfo RDB_RIEGL_WFM_SBL_FIRST = {
    /* name               */ "riegl.wfm_sbl_first",
    /* title              */ "Waveform Sample Block First",
    /* tags               */ "",
    /* description        */ "Identifier of first waveform sample block that belongs to the laser shot (0 = invalid). This is not an array index but the value of riegl.id of the block.",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_WFM_SBL_FIRST_INVALID = 0;

// Number of waveform sample blocks that belong to the laser shot
const PointAttributeInfo RDB_RIEGL_WFM_SBL_COUNT = {
    /* name               */ "riegl.wfm_sbl_count",
    /* title              */ "Waveform Sample Block Count",
    /* tags               */ "",
    /* description        */ "Number of waveform sample blocks that belong to the laser shot",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Identifier of first waveform sample value that belongs to the sample block (0 = invalid). This is not an array index but the value of riegl.id of the sample.
const PointAttributeInfo RDB_RIEGL_WFM_SDA_FIRST = {
    /* name               */ "riegl.wfm_sda_first",
    /* title              */ "Waveform Sample Data First",
    /* tags               */ "",
    /* description        */ "Identifier of first waveform sample value that belongs to the sample block (0 = invalid). This is not an array index but the value of riegl.id of the sample.",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_WFM_SDA_FIRST_INVALID = 0;

// Number of waveform sample values that belong to the sample block
const PointAttributeInfo RDB_RIEGL_WFM_SDA_COUNT = {
    /* name               */ "riegl.wfm_sda_count",
    /* title              */ "Waveform Sample Data Count",
    /* tags               */ "",
    /* description        */ "Number of waveform sample values that belong to the sample block",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Raw waveform sample value
const PointAttributeInfo RDB_RIEGL_WFM_SAMPLE_VALUE = {
    /* name               */ "riegl.wfm_sample_value",
    /* title              */ "Waveform Sample Value",
    /* tags               */ "",
    /* description        */ "Raw waveform sample value",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Zero-based index of control object type listed in riegl.control_object_catalog (meta data entry)
const PointAttributeInfo RDB_RIEGL_CONTROL_OBJECT_TYPE = {
    /* name               */ "riegl.control_object_type",
    /* title              */ "Control Object Type",
    /* tags               */ "enumeration",
    /* description        */ "Zero-based index of control object type listed in riegl.control_object_catalog (meta data entry)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 32767.0,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Model fit quality value between 0 and 1 (0 = unspecified)
const PointAttributeInfo RDB_RIEGL_MODEL_FIT_QUALITY = {
    /* name               */ "riegl.model_fit_quality",
    /* title              */ "Model Fit Quality",
    /* tags               */ "",
    /* description        */ "Model fit quality value between 0 and 1 (0 = unspecified)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Observed inclination angle of the surface in the vicinity of a control point. It is defined as the arccosine of the vertical component of the surface normal vector. Hence: angle = 0 deg: horizontal (floor); angle = 90 deg: vertical (wall); angle = 180 deg: horizontal (ceiling)
const PointAttributeInfo RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE = {
    /* name               */ "riegl.cp_surface_inclination_angle",
    /* title              */ "Surface Inclination Angle",
    /* tags               */ "",
    /* description        */ "Observed inclination angle of the surface in the vicinity of a control point. It is defined as the arccosine of the vertical component of the surface normal vector. Hence: angle = 0 deg: horizontal (floor); angle = 90 deg: vertical (wall); angle = 180 deg: horizontal (ceiling)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE_FLOOR = 0.0;
const double RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE_WALL = 90.0;
const double RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE_CEILING = 180.0;

// Tolerance angle for the inclination of the surface in the vicinity of a control point
const PointAttributeInfo RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE = {
    /* name               */ "riegl.cp_surface_inclination_tolerance_angle",
    /* title              */ "Surface Inclination Tolerance Angle",
    /* tags               */ "",
    /* description        */ "Tolerance angle for the inclination of the surface in the vicinity of a control point",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 180.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Observed northing angle (azimuth) of the surface in the vicinity of a control point. It is defined as the angle of the horizontal component of the surface normal vector wrt. north direction (measured clockwise). Hence: 0 deg: North; 90 deg: East; 180 deg: South; 270 deg: West
const PointAttributeInfo RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE = {
    /* name               */ "riegl.cp_surface_northing_angle",
    /* title              */ "Surface Northing Angle",
    /* tags               */ "",
    /* description        */ "Observed northing angle (azimuth) of the surface in the vicinity of a control point. It is defined as the angle of the horizontal component of the surface normal vector wrt. north direction (measured clockwise). Hence: 0 deg: North; 90 deg: East; 180 deg: South; 270 deg: West",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Tolerance angle for the northing angle of the surface in the vicinity of a control point
const PointAttributeInfo RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE = {
    /* name               */ "riegl.cp_surface_northing_tolerance_angle",
    /* title              */ "Surface Northing Tolerance Angle",
    /* tags               */ "",
    /* description        */ "Tolerance angle for the northing angle of the surface in the vicinity of a control point",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 180.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence values of observed XYZ point coordinates wrt. local observation coordinate system
const PointAttributeInfo RDB_RIEGL_OBS_CONFIDENCE_XYZ = {
    /* name               */ "riegl.obs_confidence_xyz",
    /* title              */ "XYZ Confidences of Observed Point",
    /* tags               */ "",
    /* description        */ "Confidence values of observed XYZ point coordinates wrt. local observation coordinate system",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence value for xy position of observed point wrt. local observation coordinate system
const PointAttributeInfo RDB_RIEGL_OBS_CONFIDENCE_XY = {
    /* name               */ "riegl.obs_confidence_xy",
    /* title              */ "XY Confidence of Observed Point",
    /* tags               */ "",
    /* description        */ "Confidence value for xy position of observed point wrt. local observation coordinate system",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence value for z position of observed point wrt. local observation coordinate system
const PointAttributeInfo RDB_RIEGL_OBS_CONFIDENCE_Z = {
    /* name               */ "riegl.obs_confidence_z",
    /* title              */ "Z Confidence of Observed Point",
    /* tags               */ "",
    /* description        */ "Confidence value for z position of observed point wrt. local observation coordinate system",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence of observation coordinates in local radial direction (range)
const PointAttributeInfo RDB_RIEGL_OBS_CONFIDENCE_RANGE = {
    /* name               */ "riegl.obs_confidence_range",
    /* title              */ "Positional Confidence of Observation in Radial Direction",
    /* tags               */ "",
    /* description        */ "Confidence of observation coordinates in local radial direction (range)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence of observation coordinates in local elevation direction (theta)
const PointAttributeInfo RDB_RIEGL_OBS_CONFIDENCE_THETA = {
    /* name               */ "riegl.obs_confidence_theta",
    /* title              */ "Positional Confidence of Observation in Theta Direction",
    /* tags               */ "",
    /* description        */ "Confidence of observation coordinates in local elevation direction (theta)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence of observation coordinates in local azimuth direction (phi)
const PointAttributeInfo RDB_RIEGL_OBS_CONFIDENCE_PHI = {
    /* name               */ "riegl.obs_confidence_phi",
    /* title              */ "Positional Confidence of Observation in Phi Direction",
    /* tags               */ "",
    /* description        */ "Confidence of observation coordinates in local azimuth direction (phi)",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Confidence value for rotation angle within plane of observed signal
const PointAttributeInfo RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT = {
    /* name               */ "riegl.obs_signal_confidence_rot",
    /* title              */ "Rotation Angle Confidence of Observed Signal",
    /* tags               */ "",
    /* description        */ "Confidence value for rotation angle within plane of observed signal",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1e-06,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Flag indicating if observation shall be used for adjustment (0 = used as verification point; 1 = used for adjustment)
const PointAttributeInfo RDB_RIEGL_USED_FOR_ADJUSTMENT = {
    /* name               */ "riegl.used_for_adjustment",
    /* title              */ "Used for Adjustment",
    /* tags               */ "enumeration",
    /* description        */ "Flag indicating if observation shall be used for adjustment (0 = used as verification point; 1 = used for adjustment)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=Used as verification point\n1=Used for adjustment\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_USED_FOR_ADJUSTMENT_VERIFICATION = 0;
const uint8_t RDB_RIEGL_USED_FOR_ADJUSTMENT_ADJUSTMENT = 1;

// ID (riegl.id) of a referenced object (0 = invalid)
const PointAttributeInfo RDB_RIEGL_REFERENCE_OBJECT_ID = {
    /* name               */ "riegl.reference_object_id",
    /* title              */ "Reference Object ID",
    /* tags               */ "",
    /* description        */ "ID (riegl.id) of a referenced object (0 = invalid)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of events in specific channel, separate counters for input and output
const PointAttributeInfo RDB_RIEGL_EVENT_INDEX = {
    /* name               */ "riegl.event_index",
    /* title              */ "Event Index",
    /* tags               */ "",
    /* description        */ "Number of events in specific channel, separate counters for input and output",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4294967295.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Width of input pulse (0 = output pulse)
const PointAttributeInfo RDB_RIEGL_EVENT_PULSE_WIDTH = {
    /* name               */ "riegl.event_pulse_width",
    /* title              */ "Event Pulse Width",
    /* tags               */ "",
    /* description        */ "Width of input pulse (0 = output pulse)",
    /* unitSymbol         */ "ms",
    /* length             */ 1,
    /* resolution         */ 0.1,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 6553.5,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Secondary Attributes"
//______________________________________________________________________________
//

// Raw range of echo
const PointAttributeInfo RDB_RIEGL_RAW_RANGE = {
    /* name               */ "riegl.raw_range",
    /* title              */ "Raw Range",
    /* tags               */ "",
    /* description        */ "Raw range of echo",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ -50000.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Raw line angle
const PointAttributeInfo RDB_RIEGL_RAW_LINE_ANGLE = {
    /* name               */ "riegl.raw_line_angle",
    /* title              */ "Raw Line Angle",
    /* tags               */ "",
    /* description        */ "Raw line angle",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Raw frame angle
const PointAttributeInfo RDB_RIEGL_RAW_FRAME_ANGLE = {
    /* name               */ "riegl.raw_frame_angle",
    /* title              */ "Raw Frame Angle",
    /* tags               */ "",
    /* description        */ "Raw frame angle",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 1e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Coarse line scan angle
const PointAttributeInfo RDB_RIEGL_LINE_ANGLE_COARSE = {
    /* name               */ "riegl.line_angle_coarse",
    /* title              */ "Line Angle Coarse",
    /* tags               */ "",
    /* description        */ "Coarse line scan angle",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Reduced line scan angle
const PointAttributeInfo RDB_RIEGL_LINE_ANGLE_REDUCED = {
    /* name               */ "riegl.line_angle_reduced",
    /* title              */ "Line Angle Reduced",
    /* tags               */ "",
    /* description        */ "Reduced line scan angle",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.025,
    /* minimumValue       */ -360.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Coarse frame scan angle
const PointAttributeInfo RDB_RIEGL_FRAME_ANGLE_COARSE = {
    /* name               */ "riegl.frame_angle_coarse",
    /* title              */ "Frame Angle Coarse",
    /* tags               */ "",
    /* description        */ "Coarse frame scan angle",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 360.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Angle of incidence of the laser beam with respect to a planar neighborhood, normal incidence gives 0 deg
const PointAttributeInfo RDB_RIEGL_ANGLE_OF_INCIDENCE = {
    /* name               */ "riegl.angle_of_incidence",
    /* title              */ "Angle of Incidence",
    /* tags               */ "",
    /* description        */ "Angle of incidence of the laser beam with respect to a planar neighborhood, normal incidence gives 0 deg",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 90.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Scan Line Index in Scan
const PointAttributeInfo RDB_RIEGL_SCAN_LINE_INDEX = {
    /* name               */ "riegl.scan_line_index",
    /* title              */ "Scan Line Index",
    /* tags               */ "",
    /* description        */ "Scan Line Index in Scan",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -2000000000.0,
    /* maximumValue       */ 2000000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Laser Shot Index in Scan Line
const PointAttributeInfo RDB_RIEGL_SHOT_INDEX_LINE = {
    /* name               */ "riegl.shot_index_line",
    /* title              */ "Laser Shot Index in Line",
    /* tags               */ "",
    /* description        */ "Laser Shot Index in Scan Line",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -2000000000.0,
    /* maximumValue       */ 2000000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Mirror facet number (0 = invalid, 1 = first facet, ...)
const PointAttributeInfo RDB_RIEGL_MIRROR_FACET = {
    /* name               */ "riegl.mirror_facet",
    /* title              */ "Mirror Facet",
    /* tags               */ "enumeration",
    /* description        */ "Mirror facet number (0 = invalid, 1 = first facet, ...)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 15.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_MIRROR_FACET_INVALID = 0;

// Scan segment number (0 = invalid, 1 = first segment, ...)
const PointAttributeInfo RDB_RIEGL_SCAN_SEGMENT = {
    /* name               */ "riegl.scan_segment",
    /* title              */ "Scan Segment",
    /* tags               */ "enumeration",
    /* description        */ "Scan segment number (0 = invalid, 1 = first segment, ...)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 15.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_SCAN_SEGMENT_INVALID = 0;

// Waveform data available for laser-shot (0 = no, 1 = yes)
const PointAttributeInfo RDB_RIEGL_WAVEFORM_AVAILABLE = {
    /* name               */ "riegl.waveform_available",
    /* title              */ "Waveform Available",
    /* tags               */ "boolean",
    /* description        */ "Waveform data available for laser-shot (0 = no, 1 = yes)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=No waveform available\n1=Waveform available\n",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_WAVEFORM_AVAILABLE_FALSE = 0;
const uint8_t RDB_RIEGL_WAVEFORM_AVAILABLE_TRUE = 1;

// 1 if the point was refraction corrected, 0 otherwise
const PointAttributeInfo RDB_RIEGL_HYDRO_REFRACTION_CORRECTED = {
    /* name               */ "riegl.hydro_refraction_corrected",
    /* title              */ "Refraction Corrected",
    /* tags               */ "boolean",
    /* description        */ "1 if the point was refraction corrected, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=No refraction correction applied\n1=Refraction correction applied\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_HYDRO_REFRACTION_CORRECTED_FALSE = 0;
const uint8_t RDB_RIEGL_HYDRO_REFRACTION_CORRECTED_TRUE = 1;

// Extinction coefficient, i.e. exponential damping coefficient usually present in water bodies. N.B.: The factor is subject to refraction correction.
const PointAttributeInfo RDB_RIEGL_EXTINCTION = {
    /* name               */ "riegl.extinction",
    /* title              */ "Extinction Coefficient",
    /* tags               */ "",
    /* description        */ "Extinction coefficient, i.e. exponential damping coefficient usually present in water bodies. N.B.: The factor is subject to refraction correction.",
    /* unitSymbol         */ "dB/m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -30.0,
    /* maximumValue       */ 30.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Amplitude from volumetric backscatter in SVB FWA
const PointAttributeInfo RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC = {
    /* name               */ "riegl.svb_amplitude_volumetric",
    /* title              */ "SVB Volumetric Backscatter Amplitude",
    /* tags               */ "",
    /* description        */ "Amplitude from volumetric backscatter in SVB FWA",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -327.68,
    /* maximumValue       */ 327.67,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Reflectance from volumetric backscatter in SVB FWA
const PointAttributeInfo RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC = {
    /* name               */ "riegl.svb_reflectance_volumetric",
    /* title              */ "SVB Volumetric Backscatter Reflectance",
    /* tags               */ "",
    /* description        */ "Reflectance from volumetric backscatter in SVB FWA",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -327.68,
    /* maximumValue       */ 327.67,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Reflectance in SVB FWA of bottom return without correcting for the attenuation of the water column
const PointAttributeInfo RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED = {
    /* name               */ "riegl.svb_reflectance_bottom_uncorrected",
    /* title              */ "SVB Bottom Reflectance Uncorrected",
    /* tags               */ "",
    /* description        */ "Reflectance in SVB FWA of bottom return without correcting for the attenuation of the water column",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -327.68,
    /* maximumValue       */ 327.67,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Attenuation of the reflectance of bottom target due to backscatter and absorption in the water column in SVB FWA
const PointAttributeInfo RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC = {
    /* name               */ "riegl.svb_attenuation_volumetric",
    /* title              */ "SVB Attenuation of Bottom Return",
    /* tags               */ "",
    /* description        */ "Attenuation of the reflectance of bottom target due to backscatter and absorption in the water column in SVB FWA",
    /* unitSymbol         */ "dB",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 100.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Bottom target amplitude is attenuated by svb_target_attenuation_coefficient_volumetric for every additional meter of target depth in the water column.
const PointAttributeInfo RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC = {
    /* name               */ "riegl.svb_target_attenuation_coefficient_volumetric",
    /* title              */ "SVB Target Attenuation Coefficient",
    /* tags               */ "",
    /* description        */ "Bottom target amplitude is attenuated by svb_target_attenuation_coefficient_volumetric for every additional meter of target depth in the water column.",
    /* unitSymbol         */ "dB/m",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 100.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Indicates surface point from SVB FWA (1 = surface)
const PointAttributeInfo RDB_RIEGL_SVB_SURFACE = {
    /* name               */ "riegl.svb_surface",
    /* title              */ "SVB Surface Point",
    /* tags               */ "boolean",
    /* description        */ "Indicates surface point from SVB FWA (1 = surface)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_SVB_SURFACE_FALSE = 0;
const uint8_t RDB_RIEGL_SVB_SURFACE_TRUE = 1;

// Indicates bottom point from SVB FWA (1 = bottom)
const PointAttributeInfo RDB_RIEGL_SVB_BOTTOM = {
    /* name               */ "riegl.svb_bottom",
    /* title              */ "SVB Bottom Point",
    /* tags               */ "boolean",
    /* description        */ "Indicates bottom point from SVB FWA (1 = bottom)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_SVB_BOTTOM_FALSE = 0;
const uint8_t RDB_RIEGL_SVB_BOTTOM_TRUE = 1;

// Path length between surface and bottom from SVB FWA
const PointAttributeInfo RDB_RIEGL_SVB_PATH_LENGTH = {
    /* name               */ "riegl.svb_path_length",
    /* title              */ "SVB Path Length",
    /* tags               */ "",
    /* description        */ "Path length between surface and bottom from SVB FWA",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.004,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 200.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 1 for all points of the first laser shot of a scan line, 0 otherwise
const PointAttributeInfo RDB_RIEGL_START_OF_SCAN_LINE = {
    /* name               */ "riegl.start_of_scan_line",
    /* title              */ "Start of Scan Line",
    /* tags               */ "boolean",
    /* description        */ "1 for all points of the first laser shot of a scan line, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_START_OF_SCAN_LINE_FALSE = 0;
const uint8_t RDB_RIEGL_START_OF_SCAN_LINE_TRUE = 1;

// 1 for all points of the last laser shot of a scan line, 0 otherwise
const PointAttributeInfo RDB_RIEGL_END_OF_SCAN_LINE = {
    /* name               */ "riegl.end_of_scan_line",
    /* title              */ "End of Scan Line",
    /* tags               */ "boolean",
    /* description        */ "1 for all points of the last laser shot of a scan line, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_END_OF_SCAN_LINE_FALSE = 0;
const uint8_t RDB_RIEGL_END_OF_SCAN_LINE_TRUE = 1;

// see LAS format specification 1.4-R13
const PointAttributeInfo RDB_RIEGL_SCAN_ANGLE = {
    /* name               */ "riegl.scan_angle",
    /* title              */ "Scan Angle",
    /* tags               */ "",
    /* description        */ "see LAS format specification 1.4-R13",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.096,
    /* minimumValue       */ -180.0,
    /* maximumValue       */ 180.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// see LAS format specification 1.4-R13
const PointAttributeInfo RDB_RIEGL_SCAN_DIRECTION = {
    /* name               */ "riegl.scan_direction",
    /* title              */ "Scan Direction",
    /* tags               */ "enumeration",
    /* description        */ "see LAS format specification 1.4-R13",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=Negative\n1=Positive\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Voxel has been collapsed with neighbor (0 = not collapsed, 1 = collapsed)
const PointAttributeInfo RDB_RIEGL_VOXEL_COLLAPSED = {
    /* name               */ "riegl.voxel_collapsed",
    /* title              */ "Voxel Collapsed with Neighbor",
    /* tags               */ "boolean",
    /* description        */ "Voxel has been collapsed with neighbor (0 = not collapsed, 1 = collapsed)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_VOXEL_COLLAPSED_FALSE = 0;
const uint8_t RDB_RIEGL_VOXEL_COLLAPSED_TRUE = 1;

// 1 if the mirror wheel rotates, 0 otherwise
const PointAttributeInfo RDB_RIEGL_LINE_SCAN_ACTIVE = {
    /* name               */ "riegl.line_scan_active",
    /* title              */ "Line Scan Active",
    /* tags               */ "",
    /* description        */ "1 if the mirror wheel rotates, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const int8_t RDB_RIEGL_LINE_SCAN_ACTIVE_FALSE = 0;
const int8_t RDB_RIEGL_LINE_SCAN_ACTIVE_TRUE = 1;

// 1 if the scanner head rotates, 0 otherwise
const PointAttributeInfo RDB_RIEGL_FRAME_SCAN_ACTIVE = {
    /* name               */ "riegl.frame_scan_active",
    /* title              */ "Frame Scan Active",
    /* tags               */ "",
    /* description        */ "1 if the scanner head rotates, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const int8_t RDB_RIEGL_FRAME_SCAN_ACTIVE_FALSE = 0;
const int8_t RDB_RIEGL_FRAME_SCAN_ACTIVE_TRUE = 1;

// 1 if the data acquisition is in progress, 0 otherwise
const PointAttributeInfo RDB_RIEGL_DATA_ACQUISITION_ACTIVE = {
    /* name               */ "riegl.data_acquisition_active",
    /* title              */ "Data Acquisition Active",
    /* tags               */ "",
    /* description        */ "1 if the data acquisition is in progress, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const int8_t RDB_RIEGL_DATA_ACQUISITION_ACTIVE_FALSE = 0;
const int8_t RDB_RIEGL_DATA_ACQUISITION_ACTIVE_TRUE = 1;

// IDs (riegl.id) of plane patches this observation refers to (0 = invalid)
const PointAttributeInfo RDB_RIEGL_PLANE_REFERENCES = {
    /* name               */ "riegl.plane_references",
    /* title              */ "Plane Patch References",
    /* tags               */ "",
    /* description        */ "IDs (riegl.id) of plane patches this observation refers to (0 = invalid)",
    /* unitSymbol         */ "",
    /* length             */ 2,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_PLANE_REFERENCES_INVALID = 0;

// Cumulative distance travelled
const PointAttributeInfo RDB_RIEGL_POF_PATH_LENGTH = {
    /* name               */ "riegl.pof_path_length",
    /* title              */ "Path Length",
    /* tags               */ "",
    /* description        */ "Cumulative distance travelled",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 1.0e-3,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4.0e6,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Position (3D) dilution of precision
const PointAttributeInfo RDB_RIEGL_POF_PDOP = {
    /* name               */ "riegl.pof_pdop",
    /* title              */ "PDOP",
    /* tags               */ "",
    /* description        */ "Position (3D) dilution of precision",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 100.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Horizontal dilution of precision
const PointAttributeInfo RDB_RIEGL_POF_HDOP = {
    /* name               */ "riegl.pof_hdop",
    /* title              */ "HDOP",
    /* tags               */ "",
    /* description        */ "Horizontal dilution of precision",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 100.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Vertical dilution of precision
const PointAttributeInfo RDB_RIEGL_POF_VDOP = {
    /* name               */ "riegl.pof_vdop",
    /* title              */ "VDOP",
    /* tags               */ "",
    /* description        */ "Vertical dilution of precision",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 100.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Age of GNSS RTK corrections (-1 = unknown)
const PointAttributeInfo RDB_RIEGL_POF_AGE_OF_CORRECTIONS = {
    /* name               */ "riegl.pof_age_of_corrections",
    /* title              */ "Age Of Corrections",
    /* tags               */ "",
    /* description        */ "Age of GNSS RTK corrections (-1 = unknown)",
    /* unitSymbol         */ "s",
    /* length             */ 1,
    /* resolution         */ 0.1,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Length of GNSS RTK baseline, i.e. the distance between the antennas of the base station and the rover
const PointAttributeInfo RDB_RIEGL_POF_BASELINE_LENGTH = {
    /* name               */ "riegl.pof_baseline_length",
    /* title              */ "Baseline Length",
    /* tags               */ "",
    /* description        */ "Length of GNSS RTK baseline, i.e. the distance between the antennas of the base station and the rover",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.1,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Solution type of GNSS
const PointAttributeInfo RDB_RIEGL_POF_SOLUTION_GNSS = {
    /* name               */ "riegl.pof_solution_gnss",
    /* title              */ "GNSS Solution",
    /* tags               */ "enumeration",
    /* description        */ "Solution type of GNSS",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 8.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "0=GNSS fix invalid\n1=GNSS fix single\n2=GNSS fix DGPS\n3=GNSS fix time only\n4=GNSS fix RTK fixed\n5=GNSS fix RTK float\n6=GNSS fix estimated\n7=GNSS fix manual\n8=GNSS fix simulated\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Total number of GNSS satellites that were used to calculate the position
const PointAttributeInfo RDB_RIEGL_POF_SATELLITES_GNSS = {
    /* name               */ "riegl.pof_satellites_gnss",
    /* title              */ "Number of GNSS Satellites",
    /* tags               */ "",
    /* description        */ "Total number of GNSS satellites that were used to calculate the position",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of GPS satellites that were used to calculate the position
const PointAttributeInfo RDB_RIEGL_POF_SATELLITES_GPS = {
    /* name               */ "riegl.pof_satellites_gps",
    /* title              */ "Number of GPS Satellites",
    /* tags               */ "",
    /* description        */ "Number of GPS satellites that were used to calculate the position",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of GLONASS satellites that were used to calculate the position
const PointAttributeInfo RDB_RIEGL_POF_SATELLITES_GLONASS = {
    /* name               */ "riegl.pof_satellites_glonass",
    /* title              */ "Number of GLONASS Satellites",
    /* tags               */ "",
    /* description        */ "Number of GLONASS satellites that were used to calculate the position",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of Beidou satellites that were used to calculate the position
const PointAttributeInfo RDB_RIEGL_POF_SATELLITES_BEIDOU = {
    /* name               */ "riegl.pof_satellites_beidou",
    /* title              */ "Number of Beidou Satellites",
    /* tags               */ "",
    /* description        */ "Number of Beidou satellites that were used to calculate the position",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of Galileo satellites that were used to calculate the position
const PointAttributeInfo RDB_RIEGL_POF_SATELLITES_GALILEO = {
    /* name               */ "riegl.pof_satellites_galileo",
    /* title              */ "Number of Galileo Satellites",
    /* tags               */ "",
    /* description        */ "Number of Galileo satellites that were used to calculate the position",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of Quasi-Zenith Satellite System (QZSS) satellites that were used to calculate the position
const PointAttributeInfo RDB_RIEGL_POF_SATELLITES_QZSS = {
    /* name               */ "riegl.pof_satellites_qzss",
    /* title              */ "Number of QZSS Satellites",
    /* tags               */ "",
    /* description        */ "Number of Quasi-Zenith Satellite System (QZSS) satellites that were used to calculate the position",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Sums of the linear point distances to the pixel corner (0: dx, 1: dy, 2: dz)
const PointAttributeInfo RDB_RIEGL_PIXEL_LINEAR_SUMS = {
    /* name               */ "riegl.pixel_linear_sums",
    /* title              */ "Linear Sums",
    /* tags               */ "",
    /* description        */ "Sums of the linear point distances to the pixel corner (0: dx, 1: dy, 2: dz)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 1e-3,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 64000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PIXEL_LINEAR_SUMS_DX = 0;
const size_t RDB_RIEGL_PIXEL_LINEAR_SUMS_DY = 1;
const size_t RDB_RIEGL_PIXEL_LINEAR_SUMS_DZ = 2;

// Sums of the square point distances to the pixel corner (0: dx*dx, 1: dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)
const PointAttributeInfo RDB_RIEGL_PIXEL_SQUARE_SUMS = {
    /* name               */ "riegl.pixel_square_sums",
    /* title              */ "Square Sums",
    /* tags               */ "",
    /* description        */ "Sums of the square point distances to the pixel corner (0: dx*dx, 1: dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)",
    /* unitSymbol         */ "m²",
    /* length             */ 6,
    /* resolution         */ 1e-6,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 5120000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_PIXEL_SQUARE_SUMS_DX_DX = 0;
const size_t RDB_RIEGL_PIXEL_SQUARE_SUMS_DY_DY = 1;
const size_t RDB_RIEGL_PIXEL_SQUARE_SUMS_DZ_DZ = 2;
const size_t RDB_RIEGL_PIXEL_SQUARE_SUMS_DX_DY = 3;
const size_t RDB_RIEGL_PIXEL_SQUARE_SUMS_DY_DZ = 4;
const size_t RDB_RIEGL_PIXEL_SQUARE_SUMS_DX_DZ = 5;

// Estimated shape of point cloud (0 = undefined, 1 = plane, 2 = line, 3 = volume)
const PointAttributeInfo RDB_RIEGL_SHAPE_ID = {
    /* name               */ "riegl.shape_id",
    /* title              */ "Point Cloud Shape",
    /* tags               */ "enumeration",
    /* description        */ "Estimated shape of point cloud (0 = undefined, 1 = plane, 2 = line, 3 = volume)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 3.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "0=Undefined\n1=Plane\n2=Line\n3=Volume\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_SHAPE_ID_UNDEFINED = 0;
const uint8_t RDB_RIEGL_SHAPE_ID_PLANE = 1;
const uint8_t RDB_RIEGL_SHAPE_ID_LINE = 2;
const uint8_t RDB_RIEGL_SHAPE_ID_VOLUME = 3;

// The plane cluster ID this point belongs to (0 = no cluster)
const PointAttributeInfo RDB_RIEGL_PLANE_CLUSTER_ID = {
    /* name               */ "riegl.plane_cluster_id",
    /* title              */ "Plane Cluster ID",
    /* tags               */ "",
    /* description        */ "The plane cluster ID this point belongs to (0 = no cluster)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4000000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// The segment ID this point belongs to (0 = no segment)
const PointAttributeInfo RDB_RIEGL_SEGMENT_ID = {
    /* name               */ "riegl.segment_id",
    /* title              */ "Segment ID",
    /* tags               */ "",
    /* description        */ "The segment ID this point belongs to (0 = no segment)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4000000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Sums of the linear point distances to the voxel corner (0: dx, 1: dy, 2: dz)
const PointAttributeInfo RDB_RIEGL_VOXEL_LINEAR_SUMS = {
    /* name               */ "riegl.voxel_linear_sums",
    /* title              */ "Linear Sums",
    /* tags               */ "",
    /* description        */ "Sums of the linear point distances to the voxel corner (0: dx, 1: dy, 2: dz)",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 2.5e-4,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 16000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_VOXEL_LINEAR_SUMS_DX = 0;
const size_t RDB_RIEGL_VOXEL_LINEAR_SUMS_DY = 1;
const size_t RDB_RIEGL_VOXEL_LINEAR_SUMS_DZ = 2;

// Sums of the square point distances to the voxel corner (0: dx*dx, 1: dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)
const PointAttributeInfo RDB_RIEGL_VOXEL_SQUARE_SUMS = {
    /* name               */ "riegl.voxel_square_sums",
    /* title              */ "Square Sums",
    /* tags               */ "",
    /* description        */ "Sums of the square point distances to the voxel corner (0: dx*dx, 1: dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)",
    /* unitSymbol         */ "m²",
    /* length             */ 6,
    /* resolution         */ 6.25e-8,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 320000000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const size_t RDB_RIEGL_VOXEL_SQUARE_SUMS_DX_DX = 0;
const size_t RDB_RIEGL_VOXEL_SQUARE_SUMS_DY_DY = 1;
const size_t RDB_RIEGL_VOXEL_SQUARE_SUMS_DZ_DZ = 2;
const size_t RDB_RIEGL_VOXEL_SQUARE_SUMS_DX_DY = 3;
const size_t RDB_RIEGL_VOXEL_SQUARE_SUMS_DY_DZ = 4;
const size_t RDB_RIEGL_VOXEL_SQUARE_SUMS_DX_DZ = 5;

// Integer coordinates of voxel corner
const PointAttributeInfo RDB_RIEGL_VOXEL_INDEX = {
    /* name               */ "riegl.voxel_index",
    /* title              */ "Voxel Index",
    /* tags               */ "",
    /* description        */ "Integer coordinates of voxel corner",
    /* unitSymbol         */ "",
    /* length             */ 3,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4294967295.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Elements 00, 11, 22, 10, 21 and 20 (in that order) of point cloud covariance matrix
const PointAttributeInfo RDB_RIEGL_COVARIANCES = {
    /* name               */ "riegl.covariances",
    /* title              */ "Point Cloud Covariances",
    /* tags               */ "",
    /* description        */ "Elements 00, 11, 22, 10, 21 and 20 (in that order) of point cloud covariance matrix",
    /* unitSymbol         */ "",
    /* length             */ 6,
    /* resolution         */ 1.0e-6,
    /* minimumValue       */ -4.5e9,
    /* maximumValue       */ 4.5e9,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of voxels this point represents (e.g. voxels combined to pixels, 0 = unknown)
const PointAttributeInfo RDB_RIEGL_VOXEL_COUNT = {
    /* name               */ "riegl.voxel_count",
    /* title              */ "Voxel Count",
    /* tags               */ "",
    /* description        */ "Number of voxels this point represents (e.g. voxels combined to pixels, 0 = unknown)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4294967295.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint32_t RDB_RIEGL_VOXEL_COUNT_UNKNOWN = 0;

// Point identifier, unique within database (0 = invalid)
const PointAttributeInfo RDB_RIEGL_ID = {
    /* name               */ "riegl.id",
    /* title              */ "PID",
    /* tags               */ "",
    /* description        */ "Point identifier, unique within database (0 = invalid)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_ID_INVALID = 0;

// Identifier of first vertex that belongs to a geometry object (e.g. polyline) (0 = invalid)
const PointAttributeInfo RDB_RIEGL_VERTEX_FIRST = {
    /* name               */ "riegl.vertex_first",
    /* title              */ "Vertex First",
    /* tags               */ "",
    /* description        */ "Identifier of first vertex that belongs to a geometry object (e.g. polyline) (0 = invalid)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 4294967295.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Number of vertices that belong to a geometry object (e.g. polyline)
const PointAttributeInfo RDB_RIEGL_VERTEX_COUNT = {
    /* name               */ "riegl.vertex_count",
    /* title              */ "Vertex Count",
    /* tags               */ "",
    /* description        */ "Number of vertices that belong to a geometry object (e.g. polyline)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Search radius for finding the control point in the scan data. This value (if valid) shall be preferred over that of the common settings when finding observation points for the respective control point.
const PointAttributeInfo RDB_RIEGL_CP_SEARCH_RADIUS = {
    /* name               */ "riegl.cp_search_radius",
    /* title              */ "Search Radius",
    /* tags               */ "",
    /* description        */ "Search radius for finding the control point in the scan data. This value (if valid) shall be preferred over that of the common settings when finding observation points for the respective control point.",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 655.35,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_CP_SEARCH_RADIUS_INVALID = 0.0;

// Maximum accepted distance of an observed control point from SOCS center. This value (if valid) shall be preferred over that of the common settings when finding observation points for the respective control point.
const PointAttributeInfo RDB_RIEGL_CP_MAXIMUM_DISTANCE = {
    /* name               */ "riegl.cp_maximum_distance",
    /* title              */ "Maximum Distance",
    /* tags               */ "",
    /* description        */ "Maximum accepted distance of an observed control point from SOCS center. This value (if valid) shall be preferred over that of the common settings when finding observation points for the respective control point.",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.1,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_CP_MAXIMUM_DISTANCE_INVALID = 0.0;

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Other Attributes"
//______________________________________________________________________________
//

// Point selected by user (0 = not selected, 1 = selected)
const PointAttributeInfo RDB_RIEGL_SELECTED = {
    /* name               */ "riegl.selected",
    /* title              */ "Selected",
    /* tags               */ "boolean",
    /* description        */ "Point selected by user (0 = not selected, 1 = selected)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_DYNAMIC,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_SELECTED_FALSE = 0;
const uint8_t RDB_RIEGL_SELECTED_TRUE = 1;

// Point visible (i.e. not hidden) in view (0 = hidden, 1 = visible)
const PointAttributeInfo RDB_RIEGL_VISIBLE = {
    /* name               */ "riegl.visible",
    /* title              */ "Visible",
    /* tags               */ "boolean",
    /* description        */ "Point visible (i.e. not hidden) in view (0 = hidden, 1 = visible)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_DYNAMIC,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_VISIBLE_FALSE = 0;
const uint8_t RDB_RIEGL_VISIBLE_TRUE = 1;

// ID of original point cloud (0 = unknown)
const PointAttributeInfo RDB_RIEGL_SOURCE_CLOUD_ID = {
    /* name               */ "riegl.source_cloud_id",
    /* title              */ "Point Cloud ID",
    /* tags               */ "",
    /* description        */ "ID of original point cloud (0 = unknown)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 10000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint16_t RDB_RIEGL_SOURCE_CLOUD_ID_UNKNOWN = 0;

// For points merged from multiple source files, the number of source files contributing to the point (0 = unknown)
const PointAttributeInfo RDB_RIEGL_SOURCE_CLOUD_COUNT = {
    /* name               */ "riegl.source_cloud_count",
    /* title              */ "Point Cloud Count",
    /* tags               */ "",
    /* description        */ "For points merged from multiple source files, the number of source files contributing to the point (0 = unknown)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 65535.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint16_t RDB_RIEGL_SOURCE_CLOUD_COUNT_UNKNOWN = 0;

// Index of point in original point cloud (0 = unknown)
const PointAttributeInfo RDB_RIEGL_SOURCE_INDEX = {
    /* name               */ "riegl.source_index",
    /* title              */ "Point Index",
    /* tags               */ "",
    /* description        */ "Index of point in original point cloud (0 = unknown)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_SOURCE_INDEX_UNKNOWN = 0;

// 0 for all points derived by standard waveform processing, 1 for additional points derived from interpolation, other enum values to be defined separately
const PointAttributeInfo RDB_RIEGL_SOURCE_INDICATOR = {
    /* name               */ "riegl.source_indicator",
    /* title              */ "Source Indicator",
    /* tags               */ "enumeration",
    /* description        */ "0 for all points derived by standard waveform processing, 1 for additional points derived from interpolation, other enum values to be defined separately",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 255.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=Standard processing\n1=Target interpolation\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_SOURCE_INDICATOR_STANDARD_PROCESSING = 0;
const uint8_t RDB_RIEGL_SOURCE_INDICATOR_TARGET_INTERPOLATION = 1;

// Marks points that belong to dynamic objects (0 = no dynamic object, 1 = dynamic object)
const PointAttributeInfo RDB_RIEGL_DYNAMIC_OBJECT_POINT = {
    /* name               */ "riegl.dynamic_object_point",
    /* title              */ "Dynamic Object Point",
    /* tags               */ "boolean",
    /* description        */ "Marks points that belong to dynamic objects (0 = no dynamic object, 1 = dynamic object)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Marks points that originate from one source file (0 = multiple source files, 1 = single source file)
const PointAttributeInfo RDB_RIEGL_SINGLE_SOURCE_POINT = {
    /* name               */ "riegl.single_source_point",
    /* title              */ "Single Source Point",
    /* tags               */ "boolean",
    /* description        */ "Marks points that originate from one source file (0 = multiple source files, 1 = single source file)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 1.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Marks points that belong to mirror objects (0 = no mirror object, 1 = mirror object)
const PointAttributeInfo RDB_RIEGL_MIRROR_OBJECT_POINT = {
    /* name               */ "riegl.mirror_object_point",
    /* title              */ "Mirror Object Point",
    /* tags               */ "boolean",
    /* description        */ "Marks points that belong to mirror objects (0 = no mirror object, 1 = mirror object)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 1 for all points in temporal vicinity of echoes from the exit aperture and corrected for the impact of the exit pane on amplitude and range, 0 otherwise
const PointAttributeInfo RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED = {
    /* name               */ "riegl.window_echo_impact_corrected",
    /* title              */ "Window Echo Impact Corrected",
    /* tags               */ "boolean",
    /* description        */ "1 for all points in temporal vicinity of echoes from the exit aperture and corrected for the impact of the exit pane on amplitude and range, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=Not impacted by exit aperture\n1=Impacted by exit aperture\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED_FALSE = 0;
const uint8_t RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED_TRUE = 1;

// 1 for all points with an echo signal amplitude below the dynamic trigger level, 0 otherwise
const PointAttributeInfo RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT = {
    /* name               */ "riegl.dyntrig_uncertain_point",
    /* title              */ "Dyntrig Uncertain Point",
    /* tags               */ "boolean",
    /* description        */ "1 for all points with an echo signal amplitude below the dynamic trigger level, 0 otherwise",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint8_t RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT_FALSE = 0;
const uint8_t RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT_TRUE = 1;

// Altitude determined based on the atmospheric pressure according to the standard atmosphere laws
const PointAttributeInfo RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL = {
    /* name               */ "riegl.barometric_height_amsl",
    /* title              */ "Barometric Altitude",
    /* tags               */ "",
    /* description        */ "Altitude determined based on the atmospheric pressure according to the standard atmosphere laws",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.01,
    /* minimumValue       */ -100.0,
    /* maximumValue       */ 10000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Distance between point and a reference surface (values less than -2000.0 mean no distance available)
const PointAttributeInfo RDB_RIEGL_DISTANCE_TO_SURFACE = {
    /* name               */ "riegl.distance_to_surface",
    /* title              */ "Distance to Surface",
    /* tags               */ "",
    /* description        */ "Distance between point and a reference surface (values less than -2000.0 mean no distance available)",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.0001,
    /* minimumValue       */ -2100.0,
    /* maximumValue       */ 2000.0,
    /* defaultValue       */ -2100.0,
    /* invalidValue       */ -2100.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const double RDB_RIEGL_DISTANCE_TO_SURFACE_LOWEST = -2000.0;

// Minimum distance to points in neighborhood
const PointAttributeInfo RDB_RIEGL_NBHD_DISTANCE_MIN = {
    /* name               */ "riegl.nbhd_distance_min",
    /* title              */ "Neighborhood Minimum Distance",
    /* tags               */ "",
    /* description        */ "Minimum distance to points in neighborhood",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Maximum distance to points in neighborhood
const PointAttributeInfo RDB_RIEGL_NBHD_DISTANCE_MAX = {
    /* name               */ "riegl.nbhd_distance_max",
    /* title              */ "Neighborhood Maximum Distance",
    /* tags               */ "",
    /* description        */ "Maximum distance to points in neighborhood",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Mean distance to points in neighborhood
const PointAttributeInfo RDB_RIEGL_NBHD_DISTANCE_MEAN = {
    /* name               */ "riegl.nbhd_distance_mean",
    /* title              */ "Neighborhood Mean Distance",
    /* tags               */ "",
    /* description        */ "Mean distance to points in neighborhood",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Median distance to points in neighborhood
const PointAttributeInfo RDB_RIEGL_NBHD_DISTANCE_MEDIAN = {
    /* name               */ "riegl.nbhd_distance_median",
    /* title              */ "Neighborhood Median Distance",
    /* tags               */ "",
    /* description        */ "Median distance to points in neighborhood",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Standard deviation of distances to points in neighborhood
const PointAttributeInfo RDB_RIEGL_NBHD_DISTANCE_STD = {
    /* name               */ "riegl.nbhd_distance_std",
    /* title              */ "Neighborhood Distance, Standard Deviation",
    /* tags               */ "",
    /* description        */ "Standard deviation of distances to points in neighborhood",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// MAD standard deviation of distances to points in neighborhood
const PointAttributeInfo RDB_RIEGL_NBHD_DISTANCE_MAD = {
    /* name               */ "riegl.nbhd_distance_mad",
    /* title              */ "Neighborhood Distance, Median Absolute Deviation",
    /* tags               */ "",
    /* description        */ "MAD standard deviation of distances to points in neighborhood",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.00025,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 50000.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_SHUFFLE,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Identifier of laser shot (0 = invalid). This is not an array index but the value of riegl.id of the laser shot.
const PointAttributeInfo RDB_RIEGL_SHOT_ID = {
    /* name               */ "riegl.shot_id",
    /* title              */ "Shot ID",
    /* tags               */ "",
    /* description        */ "Identifier of laser shot (0 = invalid). This is not an array index but the value of riegl.id of the laser shot.",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_SHOT_ID_INVALID = 0;

// Identifier of waveform sample block (0 = invalid). This is not an array index but the value of riegl.id of the block.
const PointAttributeInfo RDB_RIEGL_WFM_SBL_ID = {
    /* name               */ "riegl.wfm_sbl_id",
    /* title              */ "Waveform Sample Block ID",
    /* tags               */ "",
    /* description        */ "Identifier of waveform sample block (0 = invalid). This is not an array index but the value of riegl.id of the block.",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0e12,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ 0.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_CONSTANT,
    /* compressionOptions */ RDB_COMPRESSION_DELTA,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};
const uint64_t RDB_RIEGL_WFM_SBL_ID_INVALID = 0;

// Event fake pulse (0 = real input event, 1 = fake input event to indicate that no event was detected within set time window)
const PointAttributeInfo RDB_RIEGL_EVENT_FAKE_PULSE = {
    /* name               */ "riegl.event_fake_pulse",
    /* title              */ "Event Fake Pulse",
    /* tags               */ "boolean",
    /* description        */ "Event fake pulse (0 = real input event, 1 = fake input event to indicate that no event was detected within set time window)",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "0=real input event\n1=fake input event\n",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 1 if point is inside inner box, 0 if outside
const PointAttributeInfo RDB_RIEGL_POINT_INSIDE_AABB_BOCS = {
    /* name               */ "riegl.point_inside_aabb_bocs",
    /* title              */ "Inside Inner AABB in BOCS",
    /* tags               */ "boolean",
    /* description        */ "1 if point is inside inner box, 0 if outside",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// 1 if point is outside outer box, 0 if inside
const PointAttributeInfo RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS = {
    /* name               */ "riegl.point_outside_aabb_bocs",
    /* title              */ "Outside Outer AABB in BOCS",
    /* tags               */ "boolean",
    /* description        */ "1 if point is outside outer box, 0 if inside",
    /* unitSymbol         */ "",
    /* length             */ 1,
    /* resolution         */ 1.0,
    /* minimumValue       */ 0.0,
    /* maximumValue       */ 1.0,
    /* defaultValue       */ 0.0,
    /* invalidValue       */ RDB_NO_INVALID_VALUE,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Water surface model z uncertainty at intersection point
const PointAttributeInfo RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY = {
    /* name               */ "riegl.tpu_wsm_z_uncertainty",
    /* title              */ "Water Surface Model Z Uncertainty",
    /* tags               */ "",
    /* description        */ "Water surface model z uncertainty at intersection point",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 64.535,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Water surface model normal uncertainty at intersection point
const PointAttributeInfo RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY = {
    /* name               */ "riegl.tpu_wsm_normal_uncertainty",
    /* title              */ "Water Surface Model Normal Uncertainty",
    /* tags               */ "",
    /* description        */ "Water surface model normal uncertainty at intersection point",
    /* unitSymbol         */ "deg",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 64.535,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Uncertainties of Cartesian coordinates (0: X, 1: Y, 2: Z) resulting from uncertainty propagation based on contributing systematic and random measurement uncertainties from several sources
const PointAttributeInfo RDB_RIEGL_TPU_XYZ_UNCERTAINTIES = {
    /* name               */ "riegl.tpu_xyz_uncertainties",
    /* title              */ "XYZ Uncertainties",
    /* tags               */ "",
    /* description        */ "Uncertainties of Cartesian coordinates (0: X, 1: Y, 2: Z) resulting from uncertainty propagation based on contributing systematic and random measurement uncertainties from several sources",
    /* unitSymbol         */ "m",
    /* length             */ 3,
    /* resolution         */ 0.001,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 64.535,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Total Horizontal Uncertainty (THU) at a confidence level of 95% derived from x- and y-component of xyz uncertainties
const PointAttributeInfo RDB_RIEGL_TPU_THU = {
    /* name               */ "riegl.tpu_thu",
    /* title              */ "Total Horizontal Uncertainty",
    /* tags               */ "",
    /* description        */ "Total Horizontal Uncertainty (THU) at a confidence level of 95% derived from x- and y-component of xyz uncertainties",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 64.535,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Total Vertical Uncertainty (TVU) at a confidence level of 95% derived from z-component of xyz uncertainties
const PointAttributeInfo RDB_RIEGL_TPU_TVU = {
    /* name               */ "riegl.tpu_tvu",
    /* title              */ "Total Vertical Uncertainty",
    /* tags               */ "",
    /* description        */ "Total Vertical Uncertainty (TVU) at a confidence level of 95% derived from z-component of xyz uncertainties",
    /* unitSymbol         */ "m",
    /* length             */ 1,
    /* resolution         */ 0.001,
    /* minimumValue       */ -1.0,
    /* maximumValue       */ 64.535,
    /* defaultValue       */ -1.0,
    /* invalidValue       */ -1.0,
    /* namedValues        */ "",
    /* storageClass       */ RDB_STORAGE_VARIABLE,
    /* compressionOptions */ RDB_COMPRESSION_DEFAULT,
    /* lodSettings        */ "default",
    /* scaleFactor        */ 1.0
};

// Table of all point attribute details records
const PointAttributeInfo* const RDB_POINT_ATTRIBUTES[] = {
    &RDB_RIEGL_XYZ,
    &RDB_RIEGL_XYZ_SOCS,
    &RDB_RIEGL_XYZ_LVCS,
    &RDB_RIEGL_XYZ_MAP,
    &RDB_RIEGL_XYZ_MAP_MIN,
    &RDB_RIEGL_XYZ_MAP_MAX,
    &RDB_RIEGL_XY_MAP,
    &RDB_RIEGL_XYZ_CORRECTIONS,
    &RDB_RIEGL_RANGE,
    &RDB_RIEGL_THETA,
    &RDB_RIEGL_PHI,
    &RDB_RIEGL_SCANNER_POSITION,
    &RDB_RIEGL_DIRECTION,
    &RDB_RIEGL_DIRECTION_MEDIUM,
    &RDB_RIEGL_DIRECTION_COARSE,
    &RDB_RIEGL_SHOT_ORIGIN,
    &RDB_RIEGL_SHOT_BIAXIAL_SHIFT,
    &RDB_RIEGL_SHOT_DIRECTION,
    &RDB_RIEGL_SHOT_DIRECTION_LEVELLED,
    &RDB_RIEGL_SURFACE_NORMAL,
    &RDB_RIEGL_POINT_VECTOR,
    &RDB_RIEGL_PLANE_UP,
    &RDB_RIEGL_PLANE_COG_LINK,
    &RDB_RIEGL_PLANE_PATCH_LINK_VECTOR,
    &RDB_RIEGL_PCA_AXIS_MIN,
    &RDB_RIEGL_PCA_AXIS_MAX,
    &RDB_RIEGL_MODEL_CS_AXIS_X,
    &RDB_RIEGL_MODEL_CS_AXIS_Y,
    &RDB_RIEGL_MODEL_CS_AXIS_Z,
    &RDB_RIEGL_CYLINDER_AXIS_DIRECTION,
    &RDB_RIEGL_ACCELEROMETER,
    &RDB_RIEGL_ACCELEROMETER_RAW,
    &RDB_RIEGL_ACCELEROMETER_BIAS,
    &RDB_RIEGL_ACCELEROMETER_SCALE,
    &RDB_RIEGL_GYROSCOPE,
    &RDB_RIEGL_GYROSCOPE_RAW,
    &RDB_RIEGL_GYROSCOPE_BIAS,
    &RDB_RIEGL_GYROSCOPE_SCALE,
    &RDB_RIEGL_MAGNETIC_FIELD_SENSOR,
    &RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW,
    &RDB_RIEGL_POF_LATITUDE,
    &RDB_RIEGL_POF_LONGITUDE,
    &RDB_RIEGL_POF_HEIGHT,
    &RDB_RIEGL_POF_UNDULATION,
    &RDB_RIEGL_POF_ROLL,
    &RDB_RIEGL_POF_PITCH,
    &RDB_RIEGL_POF_YAW,
    &RDB_RIEGL_POF_COURSE,
    &RDB_RIEGL_POF_XYZ,
    &RDB_RIEGL_POF_ROLL_NED,
    &RDB_RIEGL_POF_PITCH_NED,
    &RDB_RIEGL_POF_YAW_NED,
    &RDB_RIEGL_POF_XYZ_VELOCITY,
    &RDB_RIEGL_POF_XYZ_NED_VELOCITY,
    &RDB_RIEGL_POF_XYZ_LVCS_VELOCITY,
    &RDB_RIEGL_POF_ROLL_VELOCITY,
    &RDB_RIEGL_POF_PITCH_VELOCITY,
    &RDB_RIEGL_POF_YAW_VELOCITY,
    &RDB_RIEGL_POF_VELOCITY_NED,
    &RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH,
    &RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST,
    &RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN,
    &RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST,
    &RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN,
    &RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH,
    &RDB_RIEGL_HYDRO_INTERSECTION_POINT,
    &RDB_RIEGL_HYDRO_INTERSECTION_NORMAL,
    &RDB_RIEGL_HYDRO_WSM_UNCERTAINTY,
    &RDB_RIEGL_XYZ_ACCURACIES,
    &RDB_RIEGL_ZENITH_VECTOR,
    &RDB_RIEGL_SHOT_TIMESTAMP_HR,
    &RDB_RIEGL_TIMESTAMP,
    &RDB_RIEGL_WFM_SBL_TIME_OFFSET,
    &RDB_RIEGL_WFM_ECHO_TIME_OFFSET,
    &RDB_RIEGL_PPS_TIMESTAMP_EXTERN,
    &RDB_RIEGL_PPS_TIMESTAMP_INTERN,
    &RDB_RIEGL_EVENT_TIMESTAMP,
    &RDB_RIEGL_TIMESTAMP_MIN,
    &RDB_RIEGL_TIMESTAMP_MAX,
    &RDB_RIEGL_POF_TIMESTAMP,
    &RDB_RIEGL_ACQUISITION_DATE,
    &RDB_RIEGL_REFLECTANCE,
    &RDB_RIEGL_AMPLITUDE,
    &RDB_RIEGL_INTENSITY,
    &RDB_RIEGL_GAIN,
    &RDB_RIEGL_DEVIATION,
    &RDB_RIEGL_PULSE_WIDTH,
    &RDB_RIEGL_CLASS,
    &RDB_RIEGL_RGBA,
    &RDB_RIEGL_NIR,
    &RDB_RIEGL_TEMPERATURE,
    &RDB_RIEGL_DMI_COUNT,
    &RDB_RIEGL_DMI_PATH_LENGTH_RAW,
    &RDB_RIEGL_DMI_SCALE,
    &RDB_RIEGL_MTA_ZONE,
    &RDB_RIEGL_MTA_UNCERTAIN_POINT,
    &RDB_RIEGL_FWA,
    &RDB_RIEGL_BACKGROUND_RADIATION,
    &RDB_RIEGL_BACKGROUND_RADIATION_SI,
    &RDB_RIEGL_BACKGROUND_RADIATION_INGAAS,
    &RDB_RIEGL_TEMPERATURE_ESTIMATED_SI,
    &RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS,
    &RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI,
    &RDB_RIEGL_TARGET_INDEX,
    &RDB_RIEGL_TARGET_COUNT,
    &RDB_RIEGL_TARGET_TYPE,
    &RDB_RIEGL_ECHO_FIRST,
    &RDB_RIEGL_ECHO_COUNT,
    &RDB_RIEGL_HEIGHT_CENTER,
    &RDB_RIEGL_HEIGHT_MEAN,
    &RDB_RIEGL_HEIGHT_MIN,
    &RDB_RIEGL_HEIGHT_MAX,
    &RDB_RIEGL_POINT_COUNT,
    &RDB_RIEGL_POINT_COUNT_GRID_CELL,
    &RDB_RIEGL_PCA_EXTENTS,
    &RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL,
    &RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF,
    &RDB_RIEGL_PCA_THICKNESS,
    &RDB_RIEGL_PCA_DIAMETER,
    &RDB_RIEGL_EIGENVALUE_NORM_MIN,
    &RDB_RIEGL_EIGENVALUE_NORM_MAX,
    &RDB_RIEGL_STD_DEV,
    &RDB_RIEGL_PLANE_CONFIDENCE_NORMAL,
    &RDB_RIEGL_PLANE_SLOPE_CLASS,
    &RDB_RIEGL_PLANE_OCCUPANCY,
    &RDB_RIEGL_PLANE_WIDTH,
    &RDB_RIEGL_PLANE_HEIGHT,
    &RDB_RIEGL_PLANE_COUNT,
    &RDB_RIEGL_MATCH_COUNT,
    &RDB_RIEGL_PLANE_PATCH_DISTANCE,
    &RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE,
    &RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE,
    &RDB_RIEGL_CYLINDER_RADIUS,
    &RDB_RIEGL_CYLINDER_HEIGHT,
    &RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE,
    &RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE,
    &RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE,
    &RDB_RIEGL_CYLINDER_INLIER_COUNT,
    &RDB_RIEGL_CYLINDER_OUTLIER_COUNT,
    &RDB_RIEGL_POF_ACCURACY_NORTH,
    &RDB_RIEGL_POF_ACCURACY_EAST,
    &RDB_RIEGL_POF_ACCURACY_DOWN,
    &RDB_RIEGL_POF_ACCURACY_NORTH_EAST,
    &RDB_RIEGL_POF_ACCURACY_EAST_DOWN,
    &RDB_RIEGL_POF_ACCURACY_DOWN_NORTH,
    &RDB_RIEGL_POF_ACCURACY_ROLL,
    &RDB_RIEGL_POF_ACCURACY_PITCH,
    &RDB_RIEGL_POF_ACCURACY_YAW,
    &RDB_RIEGL_WFM_SBL_CHANNEL,
    &RDB_RIEGL_WFM_SBL_MEAN,
    &RDB_RIEGL_WFM_SBL_STD_DEV,
    &RDB_RIEGL_WFM_SBL_FIRST,
    &RDB_RIEGL_WFM_SBL_COUNT,
    &RDB_RIEGL_WFM_SDA_FIRST,
    &RDB_RIEGL_WFM_SDA_COUNT,
    &RDB_RIEGL_WFM_SAMPLE_VALUE,
    &RDB_RIEGL_CONTROL_OBJECT_TYPE,
    &RDB_RIEGL_MODEL_FIT_QUALITY,
    &RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE,
    &RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE,
    &RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE,
    &RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE,
    &RDB_RIEGL_OBS_CONFIDENCE_XYZ,
    &RDB_RIEGL_OBS_CONFIDENCE_XY,
    &RDB_RIEGL_OBS_CONFIDENCE_Z,
    &RDB_RIEGL_OBS_CONFIDENCE_RANGE,
    &RDB_RIEGL_OBS_CONFIDENCE_THETA,
    &RDB_RIEGL_OBS_CONFIDENCE_PHI,
    &RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT,
    &RDB_RIEGL_USED_FOR_ADJUSTMENT,
    &RDB_RIEGL_REFERENCE_OBJECT_ID,
    &RDB_RIEGL_EVENT_INDEX,
    &RDB_RIEGL_EVENT_PULSE_WIDTH,
    &RDB_RIEGL_RAW_RANGE,
    &RDB_RIEGL_RAW_LINE_ANGLE,
    &RDB_RIEGL_RAW_FRAME_ANGLE,
    &RDB_RIEGL_LINE_ANGLE_COARSE,
    &RDB_RIEGL_LINE_ANGLE_REDUCED,
    &RDB_RIEGL_FRAME_ANGLE_COARSE,
    &RDB_RIEGL_ANGLE_OF_INCIDENCE,
    &RDB_RIEGL_SCAN_LINE_INDEX,
    &RDB_RIEGL_SHOT_INDEX_LINE,
    &RDB_RIEGL_MIRROR_FACET,
    &RDB_RIEGL_SCAN_SEGMENT,
    &RDB_RIEGL_WAVEFORM_AVAILABLE,
    &RDB_RIEGL_HYDRO_REFRACTION_CORRECTED,
    &RDB_RIEGL_EXTINCTION,
    &RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC,
    &RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC,
    &RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED,
    &RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC,
    &RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC,
    &RDB_RIEGL_SVB_SURFACE,
    &RDB_RIEGL_SVB_BOTTOM,
    &RDB_RIEGL_SVB_PATH_LENGTH,
    &RDB_RIEGL_START_OF_SCAN_LINE,
    &RDB_RIEGL_END_OF_SCAN_LINE,
    &RDB_RIEGL_SCAN_ANGLE,
    &RDB_RIEGL_SCAN_DIRECTION,
    &RDB_RIEGL_VOXEL_COLLAPSED,
    &RDB_RIEGL_LINE_SCAN_ACTIVE,
    &RDB_RIEGL_FRAME_SCAN_ACTIVE,
    &RDB_RIEGL_DATA_ACQUISITION_ACTIVE,
    &RDB_RIEGL_PLANE_REFERENCES,
    &RDB_RIEGL_POF_PATH_LENGTH,
    &RDB_RIEGL_POF_PDOP,
    &RDB_RIEGL_POF_HDOP,
    &RDB_RIEGL_POF_VDOP,
    &RDB_RIEGL_POF_AGE_OF_CORRECTIONS,
    &RDB_RIEGL_POF_BASELINE_LENGTH,
    &RDB_RIEGL_POF_SOLUTION_GNSS,
    &RDB_RIEGL_POF_SATELLITES_GNSS,
    &RDB_RIEGL_POF_SATELLITES_GPS,
    &RDB_RIEGL_POF_SATELLITES_GLONASS,
    &RDB_RIEGL_POF_SATELLITES_BEIDOU,
    &RDB_RIEGL_POF_SATELLITES_GALILEO,
    &RDB_RIEGL_POF_SATELLITES_QZSS,
    &RDB_RIEGL_PIXEL_LINEAR_SUMS,
    &RDB_RIEGL_PIXEL_SQUARE_SUMS,
    &RDB_RIEGL_SHAPE_ID,
    &RDB_RIEGL_PLANE_CLUSTER_ID,
    &RDB_RIEGL_SEGMENT_ID,
    &RDB_RIEGL_VOXEL_LINEAR_SUMS,
    &RDB_RIEGL_VOXEL_SQUARE_SUMS,
    &RDB_RIEGL_VOXEL_INDEX,
    &RDB_RIEGL_COVARIANCES,
    &RDB_RIEGL_VOXEL_COUNT,
    &RDB_RIEGL_ID,
    &RDB_RIEGL_VERTEX_FIRST,
    &RDB_RIEGL_VERTEX_COUNT,
    &RDB_RIEGL_CP_SEARCH_RADIUS,
    &RDB_RIEGL_CP_MAXIMUM_DISTANCE,
    &RDB_RIEGL_SELECTED,
    &RDB_RIEGL_VISIBLE,
    &RDB_RIEGL_SOURCE_CLOUD_ID,
    &RDB_RIEGL_SOURCE_CLOUD_COUNT,
    &RDB_RIEGL_SOURCE_INDEX,
    &RDB_RIEGL_SOURCE_INDICATOR,
    &RDB_RIEGL_DYNAMIC_OBJECT_POINT,
    &RDB_RIEGL_SINGLE_SOURCE_POINT,
    &RDB_RIEGL_MIRROR_OBJECT_POINT,
    &RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED,
    &RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT,
    &RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL,
    &RDB_RIEGL_DISTANCE_TO_SURFACE,
    &RDB_RIEGL_NBHD_DISTANCE_MIN,
    &RDB_RIEGL_NBHD_DISTANCE_MAX,
    &RDB_RIEGL_NBHD_DISTANCE_MEAN,
    &RDB_RIEGL_NBHD_DISTANCE_MEDIAN,
    &RDB_RIEGL_NBHD_DISTANCE_STD,
    &RDB_RIEGL_NBHD_DISTANCE_MAD,
    &RDB_RIEGL_SHOT_ID,
    &RDB_RIEGL_WFM_SBL_ID,
    &RDB_RIEGL_EVENT_FAKE_PULSE,
    &RDB_RIEGL_POINT_INSIDE_AABB_BOCS,
    &RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS,
    &RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY,
    &RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY,
    &RDB_RIEGL_TPU_XYZ_UNCERTAINTIES,
    &RDB_RIEGL_TPU_THU,
    &RDB_RIEGL_TPU_TVU
};
const size_t RDB_POINT_ATTRIBUTE_COUNT(
    sizeof(RDB_POINT_ATTRIBUTES) / sizeof(RDB_POINT_ATTRIBUTES[0])
);

// Table of all point attribute group names
// (same length+order as RDB_POINT_ATTRIBUTES)
const char* const RDB_POINT_ATTRIBUTE_GROUPS[] = {
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Coordinates/Vectors",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Time",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Primary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Secondary Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes",
    "Other Attributes"
};

// Point attribute tags
const char* const RDB_TAG_BOOLEAN = "boolean"; // state/flag
const char* const RDB_TAG_COLOR = "color"; // color values
const char* const RDB_TAG_DIRECTION = "direction"; // direction vector
const char* const RDB_TAG_ENUMERATION = "enumeration"; // finite set of unique values, optionally with a name
const char* const RDB_TAG_POSITION = "position"; // position vector
const char* const RDB_TAG_TIMESTAMP = "timestamp"; // timestamp of some event, relative to epoch defined in riegl.time_base
const char* const RDB_TAG_TRANSFORM = "transform"; // transform position or direction vector during import/export

// Table of all point attribute tags (alphabetical order)
const char* const RDB_POINT_ATTRIBUTE_TAGS[] = {
    RDB_TAG_BOOLEAN,
    RDB_TAG_COLOR,
    RDB_TAG_DIRECTION,
    RDB_TAG_ENUMERATION,
    RDB_TAG_POSITION,
    RDB_TAG_TIMESTAMP,
    RDB_TAG_TRANSFORM
};
const size_t RDB_POINT_ATTRIBUTE_TAG_COUNT(
    sizeof(RDB_POINT_ATTRIBUTE_TAGS) / sizeof(RDB_POINT_ATTRIBUTE_TAGS[0])
);

#endif // RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B0
