/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryUpdate.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point update query
 * \version 2015-10-14/AW: Initial version
 * \version 2017-11-24/AW: Constructors declared as "explicit" (#2825)
 * \version 2018-07-05/AW: Wrapper function bindBuffer() added
 * \version 2020-01-17/AW: Wrapper function bindMember() added (#3497)
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_QUERYUPDATE_HPP
#define RIEGL_RDB_POINTCLOUD_QUERYUPDATE_HPP

//---< INCLUDES >---------------------------------------------------------------

#include <memory>
#include <string>
#include <cstdlib>
#include <cstdint>

#include "riegl/rdb/pointcloud/dataTypes.hpp"
#include "riegl/rdb/pointcloud/pointcloudData.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< CLASS QueryUpdate >------------------------------------------------------
/*!
 * \brief Point update query
 *
 * This class can be used to update (modify) attributes of existing points.
 *
 * \see riegl::rdb::Pointcloud::update()
 *
 * \note You either must delete the query object or call close()
 *       __before__ the parent Pointcloud instance is closed/deleted!
 */
class QueryUpdate
{
public:
    /*!
     * \brief Default constructor
     *
     * Creates a null query - i.e. the query cannot be used to update points.
     *
     * \see riegl::rdb::Pointcloud::update()
     */
    explicit QueryUpdate();

    /*!
     * \brief Constructor
     *
     * Creates a query prepared for updating points.
     *
     * \note  You cannot create new QueryUpdate objects this way,
     *        use riegl::rdb::Pointcloud::update() instead.
     */
    explicit QueryUpdate(riegl::rdb::PointcloudData *pointcloud);

    /*!
     * \brief Check if query is not null
     *
     * \see valid()
     */
    operator bool() const;

    /*!
     * \brief Check if query is not null
     *
     * A null query cannot be used to update points.
     */
    bool valid() const;

    /*!
     * \brief Finish query
     *
     * Call this function when done with updating points.
     */
    void close();

    /*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a source buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * You can but don't need to define a buffer for each attribute. If
     * no buffer is defined for an attribute, it remains unchanged.
     *
     * The buffer is expected to be n*s*d bytes large, where
     * __n__ is the number of points defined in next(),
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * \note This function just stores the buffer pointer - it does
     *       __NOT__ copy the data contained in the buffer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \note This function expects a buffer for the point ID attribute.
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     */
    void bind(
        const std::string &attribute, //!< [in] attribute name
        const DataType     dataType,  //!< [in] buffer data type
        const void        *buffer,    //!< [in] buffer location
        const std::int32_t stride = 0 //!< [in] bytes between beginnings of successive elements (0: auto)
    );

    //! \copydoc bind()
    template <typename ValueType>
    void bindBuffer(
        const std::string &attribute, //!< [in] attribute name
        const ValueType   &buffer,    //!< [in] buffer (data, pointer to data, std::array or std::vector)
        const std::int32_t stride = 0 //!< [in] bytes between beginnings of successive elements (0: auto)
    )
    {
        bind(attribute, dataTypeOf(buffer), dataPointerOf(buffer), stride);
    }

    /*!
     * \brief Bind attribute buffer
     *
     * This is a variant of bindBuffer() that allows to bind a member variable
     * of an object as attribute buffer. The object can be part of a container
     * that stores the objects contiguously (e.g. std::vector, std::array) and
     * the stride is automatically derived from the object size.
     *
     * \see bindBuffer()
     */
    template <typename ObjectType, typename MemberPointer>
    void bindMember(
        const std::string  &attribute, //!< [in] attribute name
        const ObjectType   &object,    //!< [in] e.g. first object of container
        const MemberPointer member     //!< [in] object member variable pointer
    )
    {
        bindBuffer(
            attribute, object.*member,
            static_cast<std::int32_t>(sizeof(ObjectType))
        );
    }

    //! \copydoc bindMember()
    template <typename ObjectType, typename MemberPointer>
    void bindMember(
        const std::string  &attribute, //!< [in] attribute name
        const ObjectType   &object,    //!< [in] e.g. first object of container
        const MemberPointer member,    //!< [in] object member variable pointer
        const std::size_t   index      //!< [in] index for array-like object members
    )
    {
        bindBuffer(
            attribute, (object.*member)[index],
            static_cast<std::int32_t>(sizeof(ObjectType))
        );
    }

    /*!
     * \brief Update points
     *
     * Use this function to actually read the point attributes from
     * all defined buffers and update the points in the database.
     *
     * Afterwards you may re-fill the buffers or define new buffers
     * with bind() and call next() again until all points have been
     * updated.
     *
     * \remark It is assumed that the points are given in the exact same order as
     *         returned by the select query (riegl::rdb::pointcloud::QuerySelect).
     *         If the points are given in a different order, the update will still
     *         work but may take more time to finish. Of course it is not required
     *         to update all points (you don't need to provide points that you do
     *         not want to update).
     *
     * \note IEEE-754 "NaN" values contained in floating point source
     *       buffers are ignored and the attribute's default value is
     *       used instead. Furthermore IEEE-754 "Infinity" values will
     *       always cause next() to fail with error code 10414, i.e.
     *       riegl::rdb::Error::QueryAttributeValueOutOfRange.
     *
     * \note Since version 2.4.2, unit vector point attributes (e.g.
     *       "riegl.direction") are automatically normalized (divided
     *       by the vector length) on insertion. This is done for all
     *       point attributes that meet the following requirements:
     *       `length` is 3, `minimumValue` is -1, `maximumValue` is 1,
     *       `resolution` is not 1, `scaleFactor` is 1, `unitSymbol` is
     *       empty, `invalidValue` is undefined and `tags` contains the
     *       "direction" tag but not the "do-not-normalize" tag.
     *
     * \warning If you want to modify the primary point attribute (usually
     *          the point coordinates), you __must__ either read __all__ points
     *          with a select query (class QuerySelect) first and update them
     *          later, or use two different instances of riegl::rdb::Pointcloud
     *          on the same database and use one instance to read (select) and
     *          the other instance to write (update) the points. In other words:
     *          It is not allowed to alternately call next() on an instance of
     *          QuerySelect and an instance of QueryUpdate if both were started
     *          on the same Pointcloud instance and the goal is to modify the
     *          primary point attribute.
     *
     * \returns the number of points updated
     */
    std::uint32_t next(
        std::uint32_t count //!< [in] size of source buffers in terms of points
    );

private:
    struct Private;
    std::shared_ptr<Private> data;
};

}}} // namespace riegl::rdb::pointcloud

#endif // RIEGL_RDB_POINTCLOUD_QUERYUPDATE_HPP
