/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 *
 *******************************************************************************
 */

#ifndef RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
#define RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87

#include <vector>
#include <string>
#include <cstdlib>
#include <cstdint>

namespace riegl {
namespace rdb {
namespace pointcloud {

// Meta data item definition version
const std::string RDB_META_DATA_ITEMS_VERSION = "1.1.56";
const std::string RDB_META_DATA_ITEMS_DATE = "2025-07-01";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
struct MetaDataItemInfo
{
    std::string   name;
    std::string   title;
    std::string   description;
    unsigned char status;
    std::string   schema;
    std::string   example;

    operator const std::string&() const { return name; }
};

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Angular notch filter parameters for window glass "
    "echoes\",\"type\":\"object\",\"title\":\"Angular Notch "
    "Filter\",\"properties\":{\"angle\":{\"type\":\"array\",\"description\":\"Angle [de"
    "g]\",\"items\":{\"type\":\"number\"}},\"amplitude_mean\":{\"type\":\"array\",\"descr"
    "iption\":\"Mean amplitude [dB]\",\"items\":{\"type\":\"number\"}},\"range_mean\":"
    "{\"type\":\"array\",\"description\":\"Mean range [m]\",\"items\":{\"type\":\"number"
    "\"}}},\"required\":[\"angle\",\"range_mean\",\"amplitude_mean\"],\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"angle\":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],\"amp"
    "litude_mean\":[3.780913,3.780913,3.480913,3.120913,2.850913,2.720913,2."
    "680913,2.610913,2.530913,2.570913,2.570913],\"range_mean\":[0.094,0.094,"
    "0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Atmospheric "
    "parameters\",\"type\":\"object\",\"title\":\"Atmospheric Parameters\",\"properti"
    "es\":{\"pressure\":{\"type\":\"number\",\"description\":\"Pressure along "
    "measurment path "
    "[mbar]\"},\"wavelength\":{\"type\":\"number\",\"description\":\"Laser wavelength"
    " [nm]\"},\"amsl\":{\"type\":\"number\",\"description\":\"Height above mean sea "
    "level (AMSL) "
    "[m]\"},\"group_velocity\":{\"type\":\"number\",\"description\":\"Group velocity "
    "of laser beam "
    "[m/s]\"},\"temperature\":{\"type\":\"number\",\"description\":\"Temperature "
    "along measurement path "
    "[\\u00b0C]\"},\"pressure_sl\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\"},\"attenuation\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "attenuation "
    "[1/km]\"},\"rel_humidity\":{\"type\":\"number\",\"description\":\"Relative "
    "humidity along measurement path [%]\"}},\"required\":[\"temperature\",\"pres"
    "sure\",\"rel_humidity\",\"pressure_sl\",\"amsl\",\"group_velocity\",\"attenuatio"
    "n\",\"wavelength\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"pressure\":970,\"wavelength\":1550,\"amsl\":0,\"group_velocity\":299711000."
    "0,\"temperature\":7,\"pressure_sl\":970,\"attenuation\":0.028125,\"rel_humidi"
    "ty\":63}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Laser beam geometry "
    "details\",\"type\":\"object\",\"title\":\"Beam "
    "Geometry\",\"properties\":{\"beam_exit_diameter\":{\"description\":\"Beam "
    "width at exit aperture [m]\",\"type\":\"number\",\"exclusiveMinimum\":false,\""
    "minimum\":0},\"beam_divergence\":{\"description\":\"Beam divergence in far "
    "field [rad]\",\"type\":\"number\",\"exclusiveMinimum\":false,\"minimum\":0}},\"r"
    "equired\":[\"beam_exit_diameter\",\"beam_divergence\"],\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"beam_exit_diameter\":0.0072,\"beam_divergence\":0.0003}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"List of control object type definitions\",\"type\":\"objec"
    "t\",\"definitions\":{\"checkerboard2x2\":{\"description\":\"checkerboard 2 by "
    "2\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"descript"
    "ion\":\"checkerboard specific properties\",\"type\":\"object\",\"properties\":{"
    "\"square_length\":{\"description\":\"length of a square of the checkerboard"
    " in meters\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0},\"shap"
    "e\":{\"description\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"checkerb"
    "oard2x2\"]}},\"required\":[\"shape\",\"square_length\"]}]},\"circular_disk\":{\""
    "description\":\"circular disk\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/defin"
    "itions/common\"},{\"description\":\"circular disk specific properties\",\"ty"
    "pe\":\"object\",\"properties\":{\"diameter\":{\"description\":\"diameter in mete"
    "rs\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0},\"offset\":{\"de"
    "scription\":\"offset in meters, e.g. reflector constant "
    "(optional)\",\"type\":\"number\"},\"shape\":{\"description\":\"shape identifier\""
    ",\"type\":\"string\",\"enum\":[\"circular_disk\"]}},\"required\":[\"shape\",\"diame"
    "ter\"]}]},\"chevron\":{\"description\":\"chevron\",\"type\":\"object\",\"allOf\":[{"
    "\"$ref\":\"#/definitions/common\"},{\"description\":\"chevron specific proper"
    "ties\",\"type\":\"object\",\"properties\":{\"thickness\":{\"description\":\"thickn"
    "ess in meters\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0},\"o"
    "utside_edge_length\":{\"description\":\"length of the two outer edges in m"
    "eters\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0},\"shape\":{\""
    "description\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"chevron\"]}},\""
    "required\":[\"shape\",\"outside_edge_length\",\"thickness\"]}]},\"rectangle\":{"
    "\"description\":\"rectangle\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definiti"
    "ons/common\"},{\"description\":\"rectangle specific properties\",\"type\":\"ob"
    "ject\",\"properties\":{\"length\":{\"description\":\"length in meters\",\"type\":"
    "\"number\",\"exclusiveMinimum\":true,\"minimum\":0},\"width\":{\"description\":\""
    "width in meters\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0},"
    "\"shape\":{\"type\":\"string\",\"enum\":[\"rectangle\"]}},\"required\":[\"shape\",\"l"
    "ength\",\"width\"]}]},\"common\":{\"description\":\"common object properties\","
    "\"type\":\"object\",\"properties\":{\"description\":{\"description\":\"string "
    "describing the object\",\"type\":\"string\"},\"name\":{\"description\":\"unique "
    "type identifier\",\"type\":\"string\",\"minLength\":3},\"surface_type\":{\"descr"
    "iption\":\"surface material type\",\"type\":\"string\",\"enum\":[\"retro_reflect"
    "ive_foil\",\"diffuse\"]},\"shape\":{\"description\":\"shape identifier\",\"type\""
    ":\"string\",\"enum\":[\"rectangle\",\"checkerboard2x2\",\"chevron\",\"circular_di"
    "sk\",\"cylinder\",\"sphere\",\"round_corner_cube_prism\"]}},\"required\":[\"name"
    "\",\"shape\"]},\"sphere\":{\"description\":\"sphere\",\"type\":\"object\",\"allOf\":["
    "{\"$ref\":\"#/definitions/common\"},{\"description\":\"sphere specific proper"
    "ties\",\"type\":\"object\",\"properties\":{\"diameter\":{\"description\":\"diamete"
    "r in meters\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0},\"sha"
    "pe\":{\"description\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"sphere\""
    "]}},\"required\":[\"shape\",\"diameter\"]}]},\"round_corner_cube_prism\":{\"des"
    "cription\":\"round corner cube prism\",\"type\":\"object\",\"allOf\":[{\"$ref\":\""
    "#/definitions/common\"},{\"description\":\"round corner cube prism "
    "specific properties\",\"type\":\"object\",\"properties\":{\"diameter\":{\"descri"
    "ption\":\"diameter in meters\",\"type\":\"number\",\"exclusiveMinimum\":true,\"m"
    "inimum\":0},\"offset\":{\"description\":\"offset in meters, e.g. reflector "
    "constant (optional)\",\"type\":\"number\"},\"shape\":{\"description\":\"shape id"
    "entifier\",\"type\":\"string\",\"enum\":[\"round_corner_cube_prism\"]}},\"requir"
    "ed\":[\"shape\",\"diameter\"]}]},\"cylinder\":{\"description\":\"cylinder\",\"type"
    "\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"description\":\"cy"
    "linder specific properties\",\"type\":\"object\",\"properties\":{\"diameter\":{"
    "\"description\":\"diameter in meters\",\"type\":\"number\",\"exclusiveMinimum\":"
    "true,\"minimum\":0},\"shape\":{\"description\":\"shape identifier\",\"type\":\"st"
    "ring\",\"enum\":[\"cylinder\"]},\"height\":{\"description\":\"height in meters\","
    "\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0}},\"required\":[\"sha"
    "pe\",\"diameter\",\"height\"]}]}},\"title\":\"Control Object Catalog\",\"propert"
    "ies\":{\"types\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"oneOf\":[{\"$ref"
    "\":\"#/definitions/rectangle\"},{\"$ref\":\"#/definitions/checkerboard2x2\"},"
    "{\"$ref\":\"#/definitions/chevron\"},{\"$ref\":\"#/definitions/circular_disk\""
    "},{\"$ref\":\"#/definitions/cylinder\"},{\"$ref\":\"#/definitions/sphere\"},{\""
    "$ref\":\"#/definitions/round_corner_cube_prism\"}]},\"uniqueItems\":true}},"
    "\"required\":[\"types\"],\"$schema\":\"http://json-schema.org/draft-04/schema"
    "#\"}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or "
    "'diffuse')\"],\"types\":[{\"description\":\"Rectangle (60cm x "
    "30cm)\",\"length\":0.6,\"width\":0.3,\"name\":\"Rectangle "
    "60x30\",\"shape\":\"rectangle\"},{\"description\":\"Rectangle (80cm x "
    "40cm)\",\"length\":0.8,\"width\":0.4,\"name\":\"Rectangle "
    "80x40\",\"shape\":\"rectangle\"},{\"description\":\"Checkerboard (square "
    "length: 30cm)\",\"name\":\"Checkerboard2x2 30\",\"square_length\":0.3,\"shape\""
    ":\"checkerboard2x2\"},{\"description\":\"Checkerboard (square length: "
    "50cm)\",\"name\":\"Checkerboard2x2 50\",\"square_length\":0.5,\"shape\":\"checke"
    "rboard2x2\"},{\"description\":\"Chevron (a=24''; "
    "b=4'')\",\"thickness\":0.1016,\"name\":\"Chevron 24''/4''\",\"outside_edge_len"
    "gth\":0.6096,\"shape\":\"chevron\"},{\"description\":\" Circular Disk "
    "(diameter: 50cm)\",\"diameter\":0.5,\"name\":\"Circular Disk 50\",\"surface_ty"
    "pe\":\"diffuse\",\"shape\":\"circular_disk\"},{\"description\":\"flat circular "
    "reflector from retro-reflective foil\",\"name\":\"RIEGL flat reflector 50 "
    "mm\",\"surface_type\":\"retro_reflective_foil\",\"shape\":\"circular_disk\",\"di"
    "ameter\":0.05,\"offset\":0.0},{\"description\":\"flat circular reflector "
    "from retro-reflective foil\",\"name\":\"RIEGL flat reflector 100 mm\",\"surf"
    "ace_type\":\"retro_reflective_foil\",\"shape\":\"circular_disk\",\"diameter\":0"
    ".1,\"offset\":0.0},{\"description\":\"flat circular reflector from "
    "retro-reflective foil\",\"name\":\"RIEGL flat reflector 150 mm\",\"surface_t"
    "ype\":\"retro_reflective_foil\",\"shape\":\"circular_disk\",\"diameter\":0.15,\""
    "offset\":0.0},{\"description\":\"cylindrical reflector from "
    "retro-reflective foil\",\"name\":\"RIEGL cylindrical reflector 50 mm\",\"sur"
    "face_type\":\"retro_reflective_foil\",\"shape\":\"cylinder\",\"height\":0.05,\"d"
    "iameter\":0.05},{\"description\":\"cylindrical reflector from "
    "retro-reflective foil\",\"name\":\"RIEGL cylindrical reflector 100 mm\",\"su"
    "rface_type\":\"retro_reflective_foil\",\"shape\":\"cylinder\",\"height\":0.1,\"d"
    "iameter\":0.1},{\"description\":\"Sphere (diameter: 200 "
    "mm)\",\"diameter\":0.2,\"name\":\"Sphere 200 "
    "mm\",\"shape\":\"sphere\"},{\"description\":\"round corner cube "
    "prism\",\"diameter\":0.05,\"offset\":0.0,\"name\":\"Corner Cube Prism 50 "
    "mm\",\"shape\":\"round_corner_cube_prism\"}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the control object reference file\",\"type"
    "\":\"object\",\"properties\":{\"reference_file\":{\"type\":\"object\",\"descriptio"
    "n\":\"Reference to a control object "
    "file\",\"properties\":{\"file_path\":{\"description\":\"Path of the control "
    "object file relative to referring "
    "file\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"Control object "
    "file's Universally Unique Identifier (RFC 4122)\",\"type\":\"string\"}},\"re"
    "quired\":[\"file_uuid\",\"file_path\"]}},\"title\":\"Control Object Reference "
    "file\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoi"
    "nts.cpx\",\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Detection probability as a function of "
    "amplitude\",\"type\":\"object\",\"title\":\"Detection Probability\",\"properties"
    "\":{\"amplitude\":{\"type\":\"array\",\"description\":\"Amplitude [dB]\",\"items\":"
    "{\"type\":\"number\"}},\"detection_probability\":{\"type\":\"array\",\"descriptio"
    "n\":\"Detection probability [0..1]\",\"items\":{\"type\":\"number\"}}},\"require"
    "d\":[\"amplitude\",\"detection_probability\"],\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],\"detection"
    "_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the device used to acquire the point "
    "cloud\",\"type\":\"object\",\"title\":\"Device "
    "Information\",\"properties\":{\"device_build\":{\"description\":\"Device build"
    " variant\",\"type\":\"string\"},\"channel_text\":{\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"device_name\":{\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\",\"type\":\"string\"},\"channel_number\":{\"description\":\"Laser "
    "channel number (not defined or 0: single channel device)\",\"type\":\"inte"
    "ger\",\"exclusiveMinimum\":false,\"minimum\":0},\"serial_number\":{\"descripti"
    "on\":\"Device serial number (e.g. S2221234)\",\"type\":\"string\"}},\"required"
    "\":[\"device_type\",\"serial_number\"],\"$schema\":\"http://json-schema.org/dr"
    "aft-04/schema#\"}"
    ,
    /* example: */
    "{\"device_build\":\"\",\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"device_n"
    "ame\":\"Scanner 1\",\"channel_number\":0,\"serial_number\":\"S2221234\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scanner device geometry "
    "details\",\"type\":\"object\",\"title\":\"Device "
    "Geometry\",\"properties\":{\"primary\":{\"description\":\"Primary device "
    "geometry structure (mandatory)\",\"type\":\"object\",\"properties\":{\"ID\":{\"d"
    "escription\":\"Structure identifier\",\"type\":\"array\",\"maxItems\":2,\"items\""
    ":{\"type\":\"integer\"},\"minItems\":2},\"content\":{\"description\":\"Internal "
    "calibration values\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"requir"
    "ed\":[\"ID\",\"content\"]},\"secondary\":{\"description\":\"Additional device "
    "geometry structure (optional)\",\"type\":\"object\",\"properties\":{\"ID\":{\"de"
    "scription\":\"Structure identifier\",\"type\":\"array\",\"maxItems\":2,\"items\":"
    "{\"type\":\"integer\"},\"minItems\":2},\"content\":{\"description\":\"Internal "
    "calibration values\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"requir"
    "ed\":[\"ID\",\"content\"]},\"amu\":{\"description\":\"Angle Measurement "
    "Unit\",\"type\":\"object\",\"properties\":{\"lineCC\":{\"description\":\"Line "
    "Circle Count (number of LSBs per full rotation about line axis)\",\"type"
    "\":\"number\",\"exclusiveMinimum\":false,\"minimum\":0},\"frameCC\":{\"descripti"
    "on\":\"Frame Circle Count (number of LSBs per full rotation about frame "
    "axis)\",\"type\":\"number\",\"exclusiveMinimum\":false,\"minimum\":0}}}},\"requi"
    "red\":[\"primary\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[4,0],\"content\":[0]},\"secondary\":{\"ID\":[91,0],\"conten"
    "t\":[0]},\"amu\":{\"lineCC\":124000,\"frameCC\":124000}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scanner passive channel device geometry "
    "details\",\"type\":\"object\",\"title\":\"Device Geometry Passive "
    "Channel\",\"properties\":{\"primary\":{\"description\":\"Primary device "
    "geometry structure (mandatory)\",\"type\":\"object\",\"properties\":{\"ID\":{\"d"
    "escription\":\"Structure identifier\",\"type\":\"array\",\"maxItems\":2,\"items\""
    ":{\"type\":\"integer\"},\"minItems\":2},\"content\":{\"description\":\"Internal "
    "calibration values\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"requir"
    "ed\":[\"ID\",\"content\"]}},\"required\":[\"primary\"],\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[143,0],\"content\":[0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Limits of the measured values output by the device. "
    "The limits depend on the device type, measurement program and/or scan "
    "pattern.\",\"type\":\"object\",\"properties\":{\"reflectance_maximum\":{\"type\":"
    "\"number\",\"description\":\"Maximum possible reflectance in "
    "dB.\"},\"mta_zone_count_maximum\":{\"type\":\"number\",\"description\":\"Maximum"
    " number of MTA "
    "zones.\"},\"deviation_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible pulse shape deviation.\"},\"background_radiation_maximum\":{\"typ"
    "e\":\"number\",\"description\":\"Maximum possible background radiation.\"},\"a"
    "mplitude_maximum\":{\"type\":\"number\",\"description\":\"Maximum possible "
    "amplitude in "
    "dB.\"},\"echo_count_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "number of echoes a laser shot can "
    "have.\"},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible range in meters.\"},\"reflectance_minimum\":{\"type\":\"number\",\"de"
    "scription\":\"Minimum possible reflectance in "
    "dB.\"},\"range_maximum\":{\"type\":\"number\",\"description\":\"Maximum possible"
    " range in "
    "meters.\"},\"amplitude_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible amplitude in dB.\"},\"background_radiation_minimum\":{\"type\":\"nu"
    "mber\",\"description\":\"Minimum possible background radiation.\"},\"deviati"
    "on_maximum\":{\"type\":\"number\",\"description\":\"Maximum possible pulse "
    "shape deviation.\"}},\"title\":\"Device Output "
    "Limits\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"reflectance_minimum\":-100.0,\"range_maximum\":10000.0,\"amplitude_minim"
    "um\":0.0,\"reflectance_maximum\":100.0,\"mta_zone_count_maximum\":7,\"deviat"
    "ion_minimum\":-1,\"background_radiation_maximum\":0,\"background_radiation"
    "_minimum\":0,\"amplitude_maximum\":100.0,\"deviation_maximum\":32767,\"range"
    "_minimum\":2.9}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the devices used to acquire the point "
    "cloud (e.g. when merging point clouds of different "
    "devices)\",\"type\":\"array\",\"title\":\"Devices Information\",\"items\":{\"type\""
    ":\"object\",\"properties\":{\"device_build\":{\"description\":\"Device build "
    "variant\",\"type\":\"string\"},\"channel_text\":{\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"device_name\":{\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\",\"type\":\"string\"},\"channel_number\":{\"description\":\"Laser "
    "channel number (not defined or 0: single channel device)\",\"type\":\"inte"
    "ger\",\"exclusiveMinimum\":false,\"minimum\":0},\"signed\":{\"description\":\"Fl"
    "ag that is set when the original 'riegl.device' entry in the source "
    "file was correctly "
    "signed.\",\"type\":\"boolean\"},\"serial_number\":{\"description\":\"Device "
    "serial number (e.g. S2221234)\",\"type\":\"string\"}},\"required\":[\"device_t"
    "ype\",\"serial_number\"]},\"$schema\":\"http://json-schema.org/draft-04/sche"
    "ma#\"}"
    ,
    /* example: */
    "[{\"device_build\":\"\",\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"device_"
    "name\":\"Scanner 1\",\"channel_number\":0,\"signed\":false,\"serial_number\":\"S"
    "2221234\"},{\"device_build\":\"\",\"channel_text\":\"\",\"device_type\":\"VQ-1560i"
    "-DW\",\"device_name\":\"Scanner 2\",\"channel_number\":1,\"signed\":true,\"seria"
    "l_number\":\"S2222680\"},{\"device_build\":\"\",\"channel_text\":\"\",\"device_typ"
    "e\":\"VQ-1560i-DW\",\"device_name\":\"Scanner "
    "3\",\"channel_number\":2,\"signed\":true,\"serial_number\":\"S2222680\"}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Distance measuring instrument "
    "configuration\",\"type\":\"object\",\"title\":\"DMI "
    "Configuration\",\"properties\":{\"dmi_incr\":{\"description\":\"Increment in "
    "ticks, 0 means distance trigger is deactivated\",\"type\":\"integer\",\"excl"
    "usiveMinimum\":false,\"maximum\":2147483647,\"minimum\":0,\"exclusiveMaximum"
    "\":false},\"dmi_count_dir\":{\"description\":\"DMI count direction is invert"
    "ed\",\"type\":\"integer\",\"exclusiveMinimum\":false,\"maximum\":1,\"minimum\":0,"
    "\"exclusiveMaximum\":false},\"dmi_max_time\":{\"description\":\"Maximum time "
    "interval between trigger events, 0 means timeout is inactive [0.1 msec"
    "]\",\"type\":\"integer\",\"exclusiveMinimum\":false,\"maximum\":65535,\"minimum\""
    ":0,\"exclusiveMaximum\":false},\"dmi_min_time\":{\"description\":\"Minimum "
    "time interval between trigger events, 0 means rate limitation is "
    "inactive [0.1 msec]\",\"type\":\"integer\",\"exclusiveMinimum\":false,\"maximu"
    "m\":65535,\"minimum\":0,\"exclusiveMaximum\":false},\"dmi_dist_per_tick\":{\"d"
    "escription\":\"Distance per tick, just informative [m]\",\"type\":\"number\"}"
    "},\"required\":[\"dmi_count_dir\",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\""
    ",\"dmi_dist_per_tick\"],\"$schema\":\"http://json-schema.org/draft-04/schem"
    "a#\"}"
    ,
    /* example: */
    "{\"dmi_incr\":0,\"dmi_count_dir\":0,\"dmi_max_time\":100,\"dmi_min_time\":0,\"d"
    "mi_dist_per_tick\":0.00024399999529123306}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Dyntrig\",\"properties\":{\"post\":{\"type\":\"object\",\"descri"
    "ption\":\"Dyntrig post values\",\"properties\":{\"nc\":{\"type\":\"integer\",\"min"
    "imum\":0,\"description\":\"Number of entries\",\"maximum\":255},\"a\":{\"type\":\""
    "array\",\"description\":\"...\",\"maxItems\":2080,\"items\":{\"type\":\"number\"},\""
    "minItems\":1}},\"required\":[\"nc\",\"a\"]},\"scale\":{\"type\":\"number\",\"descrip"
    "tion\":\"Scale in units of dB, for calculation of line: line number N=(a"
    "mplitude-offset)/scale\"},\"tab\":{\"type\":\"array\",\"description\":\"...\",\"ma"
    "xItems\":128,\"items\":{\"description\":\"...\",\"type\":\"object\",\"properties\":"
    "{\"dec1\":{\"description\":\"Decay of post-trigger threshold after del1 in "
    "units of dB/m\",\"type\":\"number\"},\"dec2\":{\"description\":\"Rise of "
    "pre-trigger threshold after del2 in units of "
    "dB/m\",\"type\":\"number\"},\"thra2\":{\"description\":\"Pre-trigger threshold "
    "in units of dB\",\"type\":\"number\"},\"thra1\":{\"description\":\"Post-trigger "
    "threshold in units of "
    "dB\",\"type\":\"number\"},\"del2\":{\"description\":\"Length of constant "
    "pre-trigger threshold elevation in units of "
    "m\",\"type\":\"number\"},\"del1\":{\"description\":\"Length of constant "
    "post-trigger threshold elevation in units of m\",\"type\":\"number\"}},\"req"
    "uired\":[\"thra1\",\"del1\",\"dec1\",\"thra2\",\"del2\",\"dec2\"]},\"minItems\":1},\"o"
    "ffset\":{\"type\":\"number\",\"description\":\"Offset in units of dB, for "
    "calculation of line\"},\"pre\":{\"type\":\"object\",\"description\":\"Dyntrig "
    "pre values\",\"properties\":{\"nc\":{\"type\":\"integer\",\"minimum\":0,\"descript"
    "ion\":\"Number of entries\",\"maximum\":255},\"a\":{\"type\":\"array\",\"descripti"
    "on\":\"...\",\"maxItems\":2080,\"items\":{\"type\":\"number\"},\"minItems\":1}},\"re"
    "quired\":[\"nc\",\"a\"]}},\"title\":\"Dyntrig\",\"$schema\":\"http://json-schema.o"
    "rg/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"post\":{\"nc\":128,\"a\":[78,86,126,134,31]},\"scale\":1.0,\"tab\":[{\"dec1\":0"
    ".0,\"dec2\":0.0,\"thra2\":0.0,\"thra1\":0.0,\"del2\":0.0,\"del1\":0.0}],\"offset\""
    ":0.0,\"pre\":{\"nc\":128,\"a\":[78,86,126,134,31]}}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about echo "
    "files\",\"type\":\"object\",\"title\":\"Echo Information\",\"properties\":{\"echo_"
    "file\":{\"type\":\"object\",\"properties\":{\"file_extension\":{\"description\":\""
    "Echo file extension, without the leading "
    "dot\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\",\"type\":\"string\"}},\"required\":[\"file_exte"
    "nsion\"]}},\"required\":[\"echo_file\"],\"$schema\":\"http://json-schema.org/d"
    "raft-04/schema#\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_extension\":\"owp\",\"file_uuid\":\"26a03615-67c0-4bea-8"
    "fe8-c577378fe661\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details for exponential decomposition of full waveform"
    " data\",\"type\":\"object\",\"definitions\":{\"channel\":{\"type\":\"object\",\"prop"
    "erties\":{\"parameter\":{\"description\":\"parameters of the syswave "
    "exponential "
    "sum\",\"type\":\"object\",\"properties\":{\"omega\":{\"description\":\"angular "
    "frequency in Hz\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"B\":{\"descr"
    "iption\":\"imaginary part of amplitude factor in units of full-scale\",\"t"
    "ype\":\"array\",\"items\":{\"type\":\"number\"}},\"gamma\":{\"description\":\"decay "
    "in 1/second\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"A\":{\"descripti"
    "on\":\"real part of amplitude factor in units of full-scale\",\"type\":\"arr"
    "ay\",\"items\":{\"type\":\"number\"}}},\"required\":[\"A\",\"B\",\"gamma\",\"omega\"]},"
    "\"scale\":{\"description\":\"amplitude "
    "calibration\",\"type\":\"number\"},\"delay\":{\"description\":\"delay "
    "calibration in "
    "seconds\",\"type\":\"number\"},\"a_lin\":{\"description\":\"relative linear "
    "amplitude range [0..1]\",\"type\":\"number\",\"exclusiveMinimum\":false,\"maxi"
    "mum\":1,\"minimum\":0,\"exclusiveMaximum\":false}},\"required\":[\"delay\",\"sca"
    "le\",\"parameter\"]}},\"title\":\"Exponential Decomposition\",\"additionalProp"
    "erties\":false,\"patternProperties\":{\"^[0-9]+$\":{\"description\":\"one "
    "field per channel, field name is channel index\",\"$ref\":\"#/definitions/"
    "channel\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"1\":{\"parameter\":{\"omega\":[352020896.0,3647927552.0,-1977987072.0],\"B"
    "\":[-3.9,0.0,-0.3],\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\"A"
    "\":[0.9,0.3,-1.3]},\"scale\":1.0,\"delay\":3.5e-09,\"a_lin\":0.9},\"0\":{\"param"
    "eter\":{\"omega\":[352020896.0,3647927552.0,-1977987072.0],\"B\":[-3.981303"
    "2150268555,0.08622030913829803,-0.3152860999107361],\"gamma\":[-10947265"
    "28.0,-769562752.0,-848000064.0],\"A\":[0.9772450923919678,0.335433512926"
    "1017,-1.312678575515747]},\"scale\":1.0,\"delay\":3.783458418887631e-09,\"a"
    "_lin\":0.27}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"riegl.gaussian_decomposition contains information "
    "relevant for extracting calibrated amplitudes and calibrated ranges "
    "from a Gaussian decomposition of full waveform data. This information "
    "is contained in a table with five columns. Two columns are to be used "
    "as input: amplitude_lsb_low_power and amplitude_lsb_high_power. The "
    "other three columns provide the outputs. Amplitude_db gives the "
    "calibrated amplitude in the optical regime in decibels. The range "
    "offset columns provide additive range offsets, given in units of "
    "seconds, for each channel.\",\"type\":\"object\",\"title\":\"Gaussian Decompos"
    "ition\",\"properties\":{\"amplitude_db\":{\"type\":\"array\",\"items\":{\"type\":\"n"
    "umber\"}},\"range_offset_sec_high_power\":{\"type\":\"array\",\"items\":{\"type\""
    ":\"number\"}},\"range_offset_sec_low_power\":{\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\"}},\"amplitude_lsb_high_power\":{\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\"}},\"amplitude_lsb_low_power\":{\"type\":\"array\",\"items\":{\"type"
    "\":\"number\"}}},\"required\":[\"amplitude_lsb_low_power\",\"amplitude_lsb_hig"
    "h_power\",\"amplitude_db\",\"range_offset_sec_low_power\",\"range_offset_sec"
    "_high_power\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude_db\":[],\"range_offset_sec_high_power\":[],\"range_offset_sec_"
    "low_power\":[],\"amplitude_lsb_high_power\":[],\"amplitude_lsb_low_power\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Point cloud georeferencing information\",\"type\":\"object"
    "\",\"properties\":{\"crs\":{\"type\":\"object\",\"description\":\"Global "
    "Coordinate Reference System. Please note that only 3D Cartesian "
    "Coordinate Systems are "
    "allowed.\",\"properties\":{\"wkt\":{\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\",\"type\":\"string\"},\"epsg\":{\"description\":\"EPSG "
    "code\",\"type\":\"integer\",\"minimum\":0},\"name\":{\"description\":\"Coordinate "
    "reference system name\",\"type\":\"string\"}}},\"pose\":{\"type\":\"array\",\"desc"
    "ription\":\"Coordinate Transformation Matrix to transform from File "
    "Coordinate System to Global Coordinate Reference System. 4x4 matrix "
    "stored as two dimensional array, row major order.\",\"maxItems\":4,\"items"
    "\":{\"description\":\"rows\",\"type\":\"array\",\"maxItems\":4,\"items\":{\"descript"
    "ion\":\"columns\",\"type\":\"number\"},\"minItems\":4},\"minItems\":4}},\"title\":\""
    "Geo Tag\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"crs\":{\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\""
    "WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHOR"
    "ITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000000000000,AUTHOR"
    "ITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.00000000000000000000,AUTHORIT"
    "Y[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AXIS[\\\"Y\\\",EAST],AXIS[\\\"Z\\\",NO"
    "RTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\",\"epsg\":4978,\"name\":\"WGS84 Geocentr"
    "ic\"},\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,"
    "4063882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847"
    ",1138787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196"
    "],[0.0,0.0,0.0,1.0]]}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Geometric scale factor applied to point coordinates\",\""
    "type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Parameters used for georeferencing of the point cloud\""
    ",\"type\":\"object\",\"properties\":{\"trajectory_file\":{\"type\":\"object\",\"des"
    "cription\":\"Trajectory data used for georeferencing of the point "
    "cloud\",\"properties\":{\"file_extension\":{\"description\":\"Trajectory file "
    "extension, without the leading "
    "dot\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\",\"type\":\"string\"}},\"required\":[\"file_exte"
    "nsion\"]},\"socs_to_body_matrix\":{\"type\":\"array\",\"description\":\"Coordina"
    "te Transformation Matrix to transform from Scanner's Own Coordinate "
    "System to Body Coordinate System. 4x4 matrix stored as two dimensional"
    " array, row major order.\",\"maxItems\":4,\"items\":{\"description\":\"rows\",\""
    "type\":\"array\",\"maxItems\":4,\"items\":{\"description\":\"columns\",\"type\":\"nu"
    "mber\"},\"minItems\":4},\"minItems\":4},\"socs_to_rocs_matrix\":{\"type\":\"arra"
    "y\",\"description\":\"Coordinate Transformation Matrix to transform from "
    "Scanner's Own Coordinate System to Record Coordinate System. 4x4 "
    "matrix stored as two dimensional array, row major order.\",\"maxItems\":4"
    ",\"items\":{\"description\":\"rows\",\"type\":\"array\",\"maxItems\":4,\"items\":{\"d"
    "escription\":\"columns\",\"type\":\"number\"},\"minItems\":4},\"minItems\":4},\"bo"
    "dy_coordinate_system_type\":{\"type\":\"string\",\"description\":\"BODY "
    "coordinate frame (NED: North-East-Down, ENU: East-North-Up), default: "
    "NED\",\"enum\":[\"NED\",\"ENU\"]},\"trajectory_offsets\":{\"type\":\"object\",\"desc"
    "ription\":\"Correction offsets applied to the trajectory data\",\"properti"
    "es\":{\"offset_height\":{\"type\":\"number\",\"description\":\"[m]\"},\"offset_pit"
    "ch\":{\"type\":\"number\",\"description\":\"[deg]\"},\"version\":{\"type\":\"integer"
    "\",\"description\":\"Meaning of offset values and how to apply them; "
    "version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\"},\"offset_east\":{\"type\":\"number\",\"description\":\"[m]\""
    "},\"offset_yaw\":{\"type\":\"number\",\"description\":\"[deg]\"},\"offset_north\":"
    "{\"type\":\"number\",\"description\":\"[m]\"},\"offset_roll\":{\"type\":\"number\",\""
    "description\":\"[deg]\"},\"offset_time\":{\"type\":\"number\",\"description\":\"[s"
    "]\"}}}},\"title\":\"Georeferencing "
    "Parameters\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"trajectory_file\":{\"file_extension\":\"pofx\",\"file_uuid\":\"93a03615-66c0"
    "-4bea-8ff8-c577378fe660\"},\"socs_to_body_matrix\":[[-0.269827776749716,-"
    "0.723017716139738,0.635954678449952,0.0],[0.962908599449764,-0.2026051"
    "7250352,0.178208229833847,0.0],[0.0,0.660451759194338,0.7508684796801,"
    "0.0],[0.0,0.0,0.0,1.0]],\"body_coordinate_system_type\":\"NED\",\"trajector"
    "y_offsets\":{\"offset_height\":-0.2,\"offset_pitch\":0.01,\"version\":0,\"offs"
    "et_east\":0.15,\"offset_yaw\":-0.45,\"offset_north\":0.07,\"offset_roll\":0.0"
    "3,\"offset_time\":18.007}}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about GNSS processing\",\"type\":\"object\",\"proper"
    "ties\":{\"base_stations\":{\"description\":\"List of GNSS base "
    "stations\",\"type\":\"array\",\"items\":{\"description\":\"GNSS base station det"
    "ails\",\"type\":\"object\",\"properties\":{\"altitude\":{\"description\":\"Altitud"
    "e [m]\",\"type\":\"number\"},\"longitude\":{\"description\":\"Longitude "
    "[deg]\",\"type\":\"number\"},\"latitude\":{\"description\":\"Latitude [deg]\",\"ty"
    "pe\":\"number\"}},\"required\":[\"latitude\",\"longitude\",\"altitude\"]}}},\"titl"
    "e\":\"GNSS Processing "
    "Information\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"base_stations\":[{\"altitude\":379.8004150390625,\"longitude\":15.6633898"
    "22,\"latitude\":48.658391162}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Map of item names\",\"type\":\"object\",\"title\":\"Item Names"
    "\",\"additionalProperties\":false,\"patternProperties\":{\"^-?[0-9]+$\":{\"des"
    "cription\":\"One field per item, field name is item id, field value is "
    "item name\",\"type\":\"string\"}},\"$schema\":\"http://json-schema.org/draft-0"
    "4/schema#\"}"
    ,
    /* example: */
    "{\"-1\":\"Name of item with id -1\",\"47\":\"Name of item with id "
    "47\",\"1\":\"Name of item with id 1\",\"2\":\"Name of item with id 2\"}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"License keys for software "
    "features\",\"type\":\"object\",\"title\":\"Software License Keys\",\"additionalP"
    "roperties\":false,\"patternProperties\":{\"^.*$\":{\"description\":\"Each "
    "field of the object represents a feature and holds a list of license "
    "keys, where the field name is the feature "
    "name.\",\"type\":\"array\",\"items\":{\"description\":\"License key (example: "
    "'46AE032A - 39882AC4 - 9EC0A184 - 6F163D73')\",\"type\":\"string\"},\"minIte"
    "ms\":1}},\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"Full Waveform Analysis Topography with GPU support\":[\"8AB44126 - "
    "23B92250 - 16E2689F - 34EF7E7B\"],\"Full Waveform Analysis "
    "Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA - 8D2ED557\"],\"MTA "
    "resolution\":[\"468E020A - 39A922E4 - B681A184 - "
    "673E3D72\"],\"Georeferencing\":[\"46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Parameters for MTA processing\",\"type\":\"object\",\"proper"
    "ties\":{\"zone_count\":{\"type\":\"integer\",\"description\":\"Maximum number of"
    " MTA zones.\",\"maximum\":255,\"minimum\":0},\"zone_width\":{\"type\":\"number\","
    "\"description\":\"Width of a MTA zone in meter.\",\"minimum\":0},\"modulation"
    "_depth\":{\"type\":\"number\",\"description\":\"Depth of pulse position "
    "modulation in meter.\",\"minimum\":0}},\"title\":\"MTA "
    "Settings\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"zone_count\":23,\"zone_width\":149.896225,\"modulation_depth\":9.368514}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Lookup table for range correction based on raw "
    "range\",\"type\":\"object\",\"title\":\"Near Range Correction "
    "Table\",\"properties\":{\"delta\":{\"description\":\"Delta between table "
    "entries [m], first entry is at range = 0 "
    "m\",\"type\":\"number\"},\"content\":{\"description\":\"Correction value [m] to "
    "be added to the raw range\",\"type\":\"array\",\"maxItems\":2000,\"items\":{\"ty"
    "pe\":\"number\"},\"minItems\":1}},\"required\":[\"delta\",\"content\"],\"$schema\":"
    "\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"delta\":0.512,\"content\":[0.0]}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Standard deviation for range and amplitude as a "
    "function of amplitude\",\"type\":\"object\",\"title\":\"Noise Estimates\",\"prop"
    "erties\":{\"amplitude\":{\"type\":\"array\",\"description\":\"Amplitude [dB]\",\"i"
    "tems\":{\"type\":\"number\"}},\"range_sigma\":{\"type\":\"array\",\"description\":\""
    "Sigma range [m]\",\"items\":{\"type\":\"number\"}},\"amplitude_sigma\":{\"type\":"
    "\"array\",\"description\":\"Sigma amplitude [dB]\",\"items\":{\"type\":\"number\"}"
    "}},\"required\":[\"amplitude\",\"range_sigma\",\"amplitude_sigma\"],\"$schema\":"
    "\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],\"range_sig"
    "ma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,0.016,0.014"
    "],\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0."
    "432,0.39,0.354]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Notch filter parameters for window glass "
    "echoes\",\"type\":\"object\",\"title\":\"Notch Filter\",\"properties\":{\"range_ma"
    "ximum\":{\"type\":\"number\",\"description\":\"Maximum range [m]\"},\"amplitude_"
    "maximum\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Maximum amplitude"
    " [dB]\"},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum range "
    "[m]\"}},\"required\":[\"range_minimum\",\"range_maximum\",\"amplitude_maximum\""
    "],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"range_maximum\":0.2,\"amplitude_maximum\":18.0,\"range_minimum\":-0.5}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the pixels contained in the file\",\"type\""
    ":\"object\",\"definitions\":{\"pixel_size\":{\"type\":\"array\",\"description\":\"S"
    "ize of "
    "pixels.\",\"maxItems\":2,\"items\":{\"type\":\"number\",\"description\":\"Length "
    "of pixel edge [m].\",\"minimum\":0},\"minItems\":2}},\"title\":\"Pixel Informa"
    "tion\",\"properties\":{\"size\":{\"$ref\":\"#/definitions/pixel_size\",\"descrip"
    "tion\":\"Size of pixels in file coordinate system.\"},\"size_llcs\":{\"$ref\""
    ":\"#/definitions/pixel_size\",\"description\":\"Size of pixels in a locally"
    " levelled cartesian coordinate system (xy). This is only used for "
    "pixels based on a map projection.\"}},\"required\":[\"size\"],\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395],\"size_llcs\":[0.5156575"
    "252891171,0.5130835356683303]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the plane patch matching process\",\"type\""
    ":\"object\",\"definitions\":{\"file_reference\":{\"type\":\"object\",\"descriptio"
    "n\":\"Reference to a plane patch "
    "file\",\"properties\":{\"file_path\":{\"description\":\"Path of the plane "
    "patch file relative to the match "
    "file\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"Plane patch file's "
    "Universally Unique Identifier (RFC 4122)\",\"type\":\"string\"}},\"required\""
    ":[\"file_uuid\",\"file_path\"]}},\"title\":\"Plane Patch Matching\",\"propertie"
    "s\":{\"plane_patch_file_one\":{\"$ref\":\"#/definitions/file_reference\",\"des"
    "cription\":\"Reference to the plane patch file one\"},\"plane_patch_file_t"
    "wo\":{\"$ref\":\"#/definitions/file_reference\",\"description\":\"Reference to"
    " the plane patch file two\"}},\"required\":[\"plane_patch_file_one\",\"plane"
    "_patch_file_two\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"plane_patch_file_one\":{\"file_path\":\"Record009_Line001/191025_121410_"
    "Scanner_1.ptch\",\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\"},\"p"
    "lane_patch_file_two\":{\"file_path\":\"project.ptch\",\"file_uuid\":\"fa47d509"
    "-a64e-49ce-8b14-ff3130fbefa9\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Statistics about plane patches found by plane patch ex"
    "tractor\",\"type\":\"object\",\"properties\":{\"total_area\":{\"type\":\"number\",\""
    "description\":\"sum of all plane patch areas [m\\u00b2]\"},\"total_horizont"
    "al_area\":{\"type\":\"number\",\"description\":\"sum of all plane patch areas "
    "projected to horizontal plane [m\\u00b2]\"}},\"title\":\"Plane Patch "
    "Statistics\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"total_area\":14007.965,\"total_horizontal_area\":13954.601}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Settings and classes for plane slope classification\",\""
    "type\":\"object\",\"definitions\":{\"method-1\":{\"type\":\"object\",\"description"
    "\":\"Classification method 1\",\"properties\":{\"plane_classification_method"
    "\":{\"type\":\"integer\",\"description\":\"method ID (=1)\",\"maximum\":1,\"minimu"
    "m\":1},\"maximum_inclination_angle_horizontal\":{\"type\":\"number\",\"descrip"
    "tion\":\"maximum inclination angle of horizontal plane patches [deg]\",\"m"
    "aximum\":360.0,\"minimum\":-360.0}},\"required\":[\"plane_classification_met"
    "hod\",\"maximum_inclination_angle_horizontal\"]},\"method-2\":{\"type\":\"obje"
    "ct\",\"description\":\"Classification method 2\",\"properties\":{\"sloping_pla"
    "ne_classes_minimum_angle\":{\"type\":\"number\",\"description\":\"minimum "
    "inclination angle of sloping plane patches [deg]\",\"maximum\":360.0,\"min"
    "imum\":-360.0},\"sloping_plane_classes_maximum_angle\":{\"type\":\"number\",\""
    "description\":\"maximum inclination angle of sloping plane patches [deg]"
    "\",\"maximum\":360.0,\"minimum\":-360.0},\"plane_classification_method\":{\"ty"
    "pe\":\"integer\",\"description\":\"method ID (=2)\",\"maximum\":2,\"minimum\":2}}"
    ",\"required\":[\"plane_classification_method\",\"sloping_plane_classes_mini"
    "mum_angle\",\"sloping_plane_classes_maximum_angle\"]}},\"title\":\"Plane "
    "Slope Class "
    "Info\",\"properties\":{\"classes\":{\"type\":\"object\",\"description\":\"Class "
    "definition table\",\"patternProperties\":{\"^[0-9]+$\":{\"description\":\"one "
    "field per class, field name is class number, field value is class name"
    "\",\"type\":\"string\"}},\"additionalProperties\":false},\"settings\":{\"type\":\""
    "object\",\"description\":\"Classification settings, details see "
    "documentation of rdbplanes\",\"oneOf\":[{\"$ref\":\"#/definitions/method-1\"}"
    ",{\"$ref\":\"#/definitions/method-2\"}]}},\"required\":[\"settings\",\"classes\""
    "],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"classes\":{\"4\":\"sloping, pointing up and north\",\"2\":\"sloping, "
    "pointing up and east\",\"3\":\"sloping, pointing up and "
    "south\",\"10\":\"sloping, pointing down and east\",\"13\":\"sloping, pointing "
    "down and west\",\"9\":\"vertical, pointing west\",\"8\":\"vertical, pointing "
    "north\",\"12\":\"sloping, pointing down and north\",\"5\":\"sloping, pointing "
    "up and west\",\"1\":\"horizontal, pointing up\",\"7\":\"vertical, pointing "
    "south\",\"14\":\"horizontal, pointing down\",\"11\":\"sloping, pointing down "
    "and south\",\"6\":\"vertical, pointing east\"},\"settings\":{\"sloping_plane_c"
    "lasses_minimum_angle\":10.0,\"sloping_plane_classes_maximum_angle\":70.0,"
    "\"plane_classification_method\":2}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Grouping and sorting of point attributes for "
    "visualization purposes\",\"type\":\"object\",\"title\":\"Point Attribute Group"
    "s\",\"additionalProperties\":false,\"patternProperties\":{\"^.*$\":{\"descript"
    "ion\":\"Each field of the object represents a point attribute group and "
    "holds a list of point attributes, where the field name is the group "
    "name.\",\"type\":\"array\",\"items\":{\"description\":\"Point attribute full "
    "name or name pattern (perl regular expression syntax)\",\"type\":\"string\""
    "},\"minItems\":1}},\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"Secondary Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_availabl"
    "e\"],\"Time\":[\"riegl.timestamp\"],\"Other Attributes\":[\"riegl.selected\",\"r"
    "iegl.visible\"],\"Coordinates/Vectors\":[\"riegl.xyz\",\"riegl.range\",\"riegl"
    ".theta\",\"riegl.phi\"],\"Primary "
    "Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.deviation\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about point cloud "
    "files\",\"type\":\"object\",\"properties\":{\"project\":{\"description\":\"Project"
    " name\",\"type\":\"string\"},\"comments\":{\"description\":\"Comments\",\"type\":\"s"
    "tring\"},\"field_of_application\":{\"description\":\"Field of application\",\""
    "type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\""
    "BLS\",\"ILS\"]}},\"title\":\"Point Cloud "
    "Information\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"project\":\"Campaign 4\",\"comments\":\"Line "
    "3\",\"field_of_application\":\"ALS\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Estimated position and orientation information as "
    "measured by GNSS, IMU or inclination "
    "sensors\",\"type\":\"object\",\"title\":\"Pose "
    "Estimation\",\"properties\":{\"orientation\":{\"description\":\"Orientation "
    "values and orientation accuracies, measured with IMU or inclination se"
    "nsors.\",\"type\":\"object\",\"properties\":{\"roll_accuracy\":{\"description\":\""
    "Roll angle accuracy [deg]\",\"type\":\"number\",\"exclusiveMinimum\":true,\"mi"
    "nimum\":0},\"yaw\":{\"description\":\"Yaw angle about scanner Z-axis [deg]\","
    "\"type\":\"number\",\"maximum\":360,\"minimum\":-360},\"pitch_accuracy\":{\"descr"
    "iption\":\"Pitch angle accuracy [deg]\",\"type\":\"number\",\"exclusiveMinimum"
    "\":true,\"minimum\":0},\"roll\":{\"description\":\"Roll angle about scanner "
    "X-axis [deg]\",\"type\":\"number\",\"maximum\":360,\"minimum\":-360},\"pitch\":{\""
    "description\":\"Pitch angle about scanner Y-axis [deg]\",\"type\":\"number\","
    "\"maximum\":360,\"minimum\":-360},\"yaw_accuracy\":{\"description\":\"Yaw angle"
    " accuracy [deg]\",\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0}}"
    ",\"required\":[\"roll\",\"pitch\",\"yaw\",\"roll_accuracy\",\"pitch_accuracy\",\"ya"
    "w_accuracy\"]},\"barometric_height_amsl\":{\"description\":\"Altitude "
    "determined based on the atmospheric pressure according to the standard"
    " atmosphere laws "
    "[m].\",\"type\":\"number\"},\"position\":{\"description\":\"Position coordinates"
    " and position accuracy values as measured by GNSS in the specified "
    "Coordinate Reference System (CRS)\",\"type\":\"object\",\"properties\":{\"vert"
    "ical_accuracy\":{\"description\":\"Vertical accuracy [m]\",\"type\":\"number\","
    "\"exclusiveMinimum\":true,\"minimum\":0},\"coordinate_2\":{\"description\":\"Co"
    "ordinate 2 as defined by axis 2 of the specified CRS (e.g., Y, "
    "Longitude)\",\"type\":\"number\"},\"coordinate_3\":{\"description\":\"Coordinate"
    " 3 as defined by axis 3 of the specified CRS (e.g., Z, Altitude)\",\"typ"
    "e\":\"number\"},\"crs\":{\"type\":\"object\",\"description\":\"Global Coordinate "
    "Reference System\",\"properties\":{\"epsg\":{\"description\":\"EPSG "
    "code\",\"type\":\"integer\",\"minimum\":0},\"wkt\":{\"description\":\"\\\"Well-Known"
    " Text\\\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta"
    "ndards/wkt-crs)\",\"type\":\"string\"}},\"required\":[\"epsg\"]},\"horizontal_ac"
    "curacy\":{\"description\":\"Horizontal accuracy [m]\",\"type\":\"number\",\"excl"
    "usiveMinimum\":true,\"minimum\":0},\"coordinate_1\":{\"description\":\"Coordin"
    "ate 1 as defined by axis 1 of the specified CRS (e.g., X, Latitude)\",\""
    "type\":\"number\"}},\"required\":[\"coordinate_1\",\"coordinate_2\",\"coordinate"
    "_3\",\"horizontal_accuracy\",\"vertical_accuracy\",\"crs\"]}},\"required\":[\"or"
    "ientation\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"orientation\":{\"roll_accuracy\":0.009433783936875745,\"yaw\":101.8729363"
    "0292045,\"pitch_accuracy\":0.009433783936875745,\"roll\":3.14743073066123,"
    "\"pitch\":1.509153024827064,\"yaw_accuracy\":1.0094337839368757},\"barometr"
    "ic_height_amsl\":386.7457796227932,\"position\":{\"vertical_accuracy\":1.33"
    "14999341964722,\"coordinate_2\":15.645033406,\"coordinate_3\":362.71249389"
    "64844,\"crs\":{\"epsg\":4979,\"wkt\":\"GEOGCS[\\\"WGS84 / Geographic\\\",DATUM[\\\""
    "WGS84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\"
    "\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000"
    "000000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Degree\\\",0.017453292"
    "51994329547,AUTHORITY[\\\"EPSG\\\",\\\"9102\\\"]],AXIS[\\\"Latitude\\\",NORTH],AXI"
    "S[\\\"Longitude\\\",EAST],AUTHORITY[\\\"EPSG\\\",\\\"4979\\\"]]\"},\"horizontal_accu"
    "racy\":0.810699999332428,\"coordinate_1\":48.655799473}}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details on position and orientation sensors\",\"type\":\"o"
    "bject\",\"definitions\":{\"vector\":{\"type\":\"array\",\"maxItems\":3,\"items\":{\""
    "type\":\"number\",\"description\":\"Index 0=X, 1=Y, 2=Z "
    "component\"},\"minItems\":3}},\"title\":\"Pose Sensors\",\"properties\":{\"accel"
    "erometer\":{\"type\":\"object\",\"description\":\"Accelerometer details\",\"prop"
    "erties\":{\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensit"
    "ive Z axis of sensor at frame angle = "
    "0\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X"
    " axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\""
    "#/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionl"
    "ess\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive"
    " Y axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement "
    "values\"},\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor"
    " origin in SOCS [m] at frame angle = "
    "0\"},\"unit\":{\"type\":\"number\",\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in 9.81 m/s\\u00b2\",\"exclusiveMinimum\":true,\""
    "minimum\":0}},\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"o"
    "rigin\",\"relative_nonlinearity\"]},\"gyroscope\":{\"type\":\"object\",\"descrip"
    "tion\":\"Gyroscope details\",\"properties\":{\"z_axis\":{\"$ref\":\"#/definition"
    "s/vector\",\"description\":\"Sensitive Z axis of sensor at frame angle = "
    "0\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X"
    " axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\""
    "#/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionl"
    "ess\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive"
    " Y axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement "
    "values\"},\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor"
    " origin in SOCS [m] at frame angle = "
    "0\"},\"unit\":{\"type\":\"number\",\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in rad/s\",\"exclusiveMinimum\":true,\"minimum\":"
    "0}},\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"r"
    "elative_nonlinearity\"]},\"magnetic_field_sensor\":{\"type\":\"object\",\"desc"
    "ription\":\"Magnetic Field Sensor details\",\"properties\":{\"fixed\":{\"$ref\""
    ":\"#/definitions/vector\",\"description\":\"Distortion of magnetic field "
    "caused by non-rotating scanner part\"},\"z_axis\":{\"$ref\":\"#/definitions/"
    "vector\",\"description\":\"Sensitive Z axis of sensor at frame angle = "
    "0\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X"
    " axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\""
    "#/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionl"
    "ess\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive"
    " Y axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement "
    "values\"},\"unit\":{\"type\":\"number\",\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in nT\",\"exclusiveMinimum\":true,\"minimum\":0}}"
    ",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"fixed\",\"relat"
    "ive_nonlinearity\"]}},\"required\":[\"gyroscope\",\"accelerometer\",\"magnetic"
    "_field_sensor\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"accelerometer\":{\"z_axis\":[1.639882206916809,15166.744140625,-116.997"
    "42889404297],\"x_axis\":[-15008.123046875,56.956390380859375,-60.5175666"
    "809082],\"relative_nonlinearity\":[0.0,0.0,0.0],\"y_axis\":[-7.02728891372"
    "6807,-44.12333679199219,14952.3701171875],\"offset\":[-733.3636474609375"
    ",58.969032287597656,1060.2550048828125],\"origin\":[0.026900000870227814"
    ",-0.03999999910593033,-0.08950000256299973],\"unit\":6.666666740784422e-"
    "05},\"gyroscope\":{\"z_axis\":[0.555869996547699,119.22135162353516,0.4675"
    "85027217865],\"x_axis\":[-121.195556640625,0.8219714164733887,0.23130317"
    "03233719],\"relative_nonlinearity\":[2.888176311444113e-07,1.06274164579"
    "645e-07,-1.7186295080634935e-39],\"y_axis\":[-0.440765917301178,-0.78973"
    "99663925171,119.5894775390625],\"offset\":[-50.92609786987305,146.156433"
    "10546875,62.4327278137207],\"origin\":[0.026900000870227814,-0.039999999"
    "10593033,-0.08950000256299973],\"unit\":0.00014544410805683583},\"magneti"
    "c_field_sensor\":{\"fixed\":[-1576.010498046875,1596.081787109375,0.0],\"z"
    "_axis\":[0.00041987866279669106,7.876977906562388e-05,0.011407104320824"
    "146],\"x_axis\":[-0.011162743903696537,-2.315962774446234e-05,0.00016818"
    "844596855342],\"relative_nonlinearity\":[0.0,0.0,0.0],\"y_axis\":[0.000278"
    "88521435670555,-0.011427424848079681,-5.204829722060822e-05],\"offset\":"
    "[-23812.052734375,5606.57666015625,2493.28125],\"unit\":91.7431182861328"
    "1}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Laser pulse position modulation used for MTA "
    "resolution\",\"type\":\"object\",\"title\":\"Pulse Position Modulation\",\"prope"
    "rties\":{\"pulse_interval\":{\"type\":\"array\",\"description\":\"Explicit table"
    " of the pulse position modulation used for MTA resolution. Table gives"
    " times between successive laser pulses in seconds.\",\"items\":{\"type\":\"n"
    "umber\",\"minimum\":0}},\"length\":{\"type\":\"integer\",\"description\":\"Length "
    "of code\",\"maximum\":255,\"minimum\":0},\"num_mod_ampl\":{\"type\":\"integer\",\""
    "description\":\"Number of different modulation amplitudes (2: binary mod"
    "ulation)\",\"maximum\":255,\"minimum\":0},\"phase_step\":{\"type\":\"integer\",\"d"
    "escription\":\"Step width in phase of modulation code from line to line\""
    ",\"maximum\":255,\"minimum\":0},\"code_phase_mode\":{\"type\":\"integer\",\"descr"
    "iption\":\"0: no synchronization, 1: toggle between 2 phases, 2: "
    "increment with phase_increment\",\"maximum\":255,\"minimum\":0}},\"required\""
    ":[\"length\",\"num_mod_ampl\",\"pulse_interval\"],\"$schema\":\"http://json-sch"
    "ema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"pulse_interval\":[2.759375e-06,2.759375e-06,2.759375e-06,2.759375e-06"
    ",2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.82"
    "1875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e"
    "-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2"
    ".759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.8218"
    "75e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.821875e-0"
    "6],\"length\":31,\"num_mod_ampl\":2,\"phase_step\":5,\"code_phase_mode\":2}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Statistics about target distance wrt. SOCS "
    "origin\",\"type\":\"object\",\"title\":\"Range Statistics\",\"properties\":{\"mini"
    "mum\":{\"type\":\"number\",\"description\":\"Minimum "
    "value\"},\"std_dev\":{\"type\":\"number\",\"description\":\"Standard "
    "deviation\"},\"average\":{\"type\":\"number\",\"description\":\"Average "
    "value\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum value\"}},\"re"
    "quired\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"minimum\":0.919,\"std_dev\":24.349,\"average\":15.49738,\"maximum\":574.35}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Receiver Internals\",\"type\":\"object\",\"definitions\":{\"fp"
    "_table_row\":{\"type\":\"array\",\"maxItems\":2048,\"minItems\":1,\"items\":{\"typ"
    "e\":\"number\"}},\"fp\":{\"type\":\"object\",\"description\":\"Fingerprint values\""
    ",\"properties\":{\"s\":{\"type\":\"array\",\"maxItems\":256,\"items\":{\"type\":\"arr"
    "ay\",\"maxItems\":4096,\"items\":{\"type\":\"number\"},\"minItems\":1},\"minItems\""
    ":1},\"w\":{\"type\":\"array\",\"maxItems\":256,\"items\":{\"type\":\"array\",\"maxIte"
    "ms\":5,\"items\":{\"type\":\"number\"},\"minItems\":5},\"minItems\":1}},\"required"
    "\":[\"s\",\"w\"]},\"fp_table\":{\"type\":\"object\",\"properties\":{\"ch\":{\"type\":\"i"
    "nteger\",\"description\":\"channel number\",\"max\":255,\"min\":0},\"ny\":{\"type\""
    ":\"integer\",\"description\":\"number of y entries\",\"max\":2048,\"min\":1},\"tc"
    "\":{\"type\":\"integer\",\"description\":\"table data type code\",\"max\":255,\"mi"
    "n\":0},\"tv\":{\"type\":\"array\",\"maxItems\":2048,\"minItems\":1,\"items\":{\"oneO"
    "f\":[{\"$ref\":\"#/definitions/fp_table_row\"},{\"type\":\"number\"}]}},\"nx\":{\""
    "type\":\"integer\",\"description\":\"number of x entries\",\"max\":2048,\"min\":1"
    "}},\"required\":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv\"],\"desription\":\"scanner "
    "internal data\"}},\"title\":\"Receiver Internals\",\"properties\":{\"ns\":{\"typ"
    "e\":\"integer\",\"minimum\":0,\"description\":\"Number of samples\",\"maximum\":4"
    "095},\"mw\":{\"type\":\"number\",\"minimum\":0,\"exclusiveMinimum\":true,\"descri"
    "ption\":\"Maximum weight\",\"maximum\":4095},\"t\":{\"type\":\"object\",\"patternP"
    "roperties\":{\"^[0-9]+$\":{\"description\":\"one field per channel, field "
    "name is channel index\",\"$ref\":\"#/definitions/fp\"}},\"additionalProperti"
    "es\":false},\"nt\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"Number of"
    " traces\",\"maximum\":255},\"tbl\":{\"type\":\"array\",\"description\":\"various "
    "internal data\",\"items\":{\"$ref\":\"#/definitions/fp_table\"},\"minItems\":1}"
    ",\"sr\":{\"type\":\"number\",\"minimum\":0,\"exclusiveMinimum\":true,\"descriptio"
    "n\":\"Sample rate "
    "[Hz]\"},\"si\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Start index (h"
    "w_start)\",\"maximum\":255},\"a\":{\"type\":\"array\",\"description\":\"Amplitude "
    "[dB]\",\"maxItems\":256,\"items\":{\"type\":\"number\"},\"minItems\":1},\"ex\":{\"ty"
    "pe\":\"object\",\"description\":\"DEPRECATED, use "
    "'riegl.exponential_decomposition' "
    "instead\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"ns\":400,\"mw\":31,\"t\":{\"1\":{\"s\":[[1.23,4.56],[7.89,0.12]],\"w\":[[78,86,"
    "126,134,31],[78,86,126,134,31]]},\"0\":{\"s\":[[1.23,4.56],[7.89,0.12]],\"w"
    "\":[[78,86,126,134,31],[78,86,126,134,31]]}},\"nt\":128,\"tbl\":[{\"ch\":0,\"n"
    "y\":2,\"tc\":1,\"tv\":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.5]],\"nx\":5}],\"sr\":795"
    "9997000.0,\"si\":48,\"a\":[-1.55]}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Lookup table for reflectance calculation based on "
    "amplitude and range\",\"type\":\"object\",\"title\":\"Reflectance Calculation "
    "Table\",\"properties\":{\"delta\":{\"description\":\"Delta between table "
    "entries [m], first entry is at range = 0 "
    "m\",\"type\":\"number\"},\"content\":{\"description\":\"Correction value [dB] to"
    " be added to the amplitude\",\"type\":\"array\",\"maxItems\":2000,\"items\":{\"t"
    "ype\":\"number\"},\"minItems\":1}},\"required\":[\"delta\",\"content\"],\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"delta\":0.150918,\"content\":[-33.01]}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Range-dependent and scan-angle-dependent correction of"
    " reflectance reading\",\"type\":\"object\",\"title\":\"Near-range reflectance "
    "correction\",\"properties\":{\"ranges_m\":{\"type\":\"array\",\"description\":\"Ra"
    "nge [m]\",\"items\":{\"type\":\"number\"}},\"line_angles_deg\":{\"type\":\"array\","
    "\"description\":\"Angle [deg]\",\"items\":{\"type\":\"number\"}},\"reflectance_co"
    "rrection_db\":{\"type\":\"array\",\"description\":\"Near range reflectance "
    "correction in dB as a function of range over "
    "angle\",\"items\":{\"description\":\"rows (each array corresponds to a "
    "range)\",\"type\":\"array\",\"items\":{\"description\":\"columns (each value "
    "corresponds to an angle)\",\"type\":\"number\"}}}},\"required\":[\"ranges_m\",\""
    "line_angles_deg\",\"reflectance_correction_db\"],\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"ranges_m\":[0.0,1.0,2.0,3.0],\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2"
    ".0,2.5,3.0,3.5,4.0],\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,"
    "0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0."
    "8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,"
    "0.1,0.05,0.01]]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about RiLOC "
    "processing\",\"type\":\"object\",\"title\":\"RiLOC Processing "
    "Information\",\"properties\":{\"trajectory\":{\"description\":\"Trajectory "
    "information and statistics.\",\"type\":\"object\",\"properties\":{}},\"optimiz"
    "er\":{\"description\":\"Optimization result information and statistics.\",\""
    "type\":\"object\",\"properties\":{}},\"georeferencing\":{\"description\":\"Inter"
    "nal georeferencing parameters.\",\"type\":\"object\",\"properties\":{}},\"plan"
    "es\":{\"description\":\"Information on plane correspondences and used "
    "extraction/matching templates.\",\"type\":\"object\",\"properties\":{}}},\"req"
    "uired\":[\"trajectory\",\"optimizer\",\"georeferencing\"],\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"trajectory\":{},\"optimizer\":{},\"georeferencing\":{},\"planes\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scan pattern "
    "description\",\"definitions\":{\"program\":{\"description\":\"Measurement prog"
    "ram\",\"type\":\"object\",\"properties\":{\"laser_prr\":{\"description\":\"Laser "
    "Pulse Repetition Rate [Hz]\",\"type\":\"number\",\"exclusiveMinimum\":false,\""
    "minimum\":0},\"name\":{\"description\":\"Name of measurement "
    "program\",\"type\":\"string\"}},\"required\":[\"name\"]}},\"title\":\"Scan "
    "Pattern\",\"properties\":{\"rectangular\":{\"description\":\"Rectangular Field"
    " Of View Scan Pattern\",\"type\":\"object\",\"properties\":{\"theta_stop\":{\"de"
    "scription\":\"Stop theta angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":"
    "180.0,\"minimum\":0.0},\"phi_stop\":{\"description\":\"Stop phi angle in SOCS"
    " [deg]\",\"type\":\"number\",\"maximum\":720.0,\"minimum\":0.0},\"program\":{\"$re"
    "f\":\"#/definitions/program\"},\"theta_start\":{\"description\":\"Start theta "
    "angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":180.0,\"minimum\":0.0},\"p"
    "hi_start\":{\"description\":\"Start phi angle in SOCS [deg]\",\"type\":\"numbe"
    "r\",\"maximum\":360.0,\"minimum\":0.0},\"theta_increment\":{\"description\":\"In"
    "crement of theta angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0,\""
    "minimum\":0.0},\"phi_increment\":{\"description\":\"Increment of phi angle "
    "in SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0,\"minimum\":0.0}},\"require"
    "d\":[\"phi_start\",\"phi_stop\",\"phi_increment\",\"theta_start\",\"theta_stop\","
    "\"theta_increment\"]},\"line\":{\"description\":\"Line Scan "
    "Pattern\",\"type\":\"object\",\"properties\":{\"start\":{\"description\":\"Start "
    "angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":360.0,\"minimum\":0.0},\"i"
    "ncrement\":{\"description\":\"Increment of angle in SOCS [deg]\",\"type\":\"nu"
    "mber\",\"maximum\":90.0,\"minimum\":0.0},\"stop\":{\"description\":\"Stop angle "
    "in SOCS [deg]\",\"type\":\"number\",\"maximum\":720.0,\"minimum\":0.0},\"program"
    "\":{\"$ref\":\"#/definitions/program\"}},\"required\":[\"start\",\"stop\",\"increm"
    "ent\"]},\"segments\":{\"description\":\"Segmented Line Scan Pattern\",\"type\":"
    "\"object\",\"properties\":{\"program\":{\"$ref\":\"#/definitions/program\"},\"lis"
    "t\":{\"type\":\"array\",\"items\":{\"description\":\"Line Scan "
    "Segment\",\"type\":\"object\",\"properties\":{\"start\":{\"description\":\"Start "
    "angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":360.0,\"minimum\":0.0},\"i"
    "ncrement\":{\"description\":\"Increment of angle in SOCS [deg]\",\"type\":\"nu"
    "mber\",\"maximum\":90.0,\"minimum\":0.0},\"stop\":{\"description\":\"Stop angle "
    "in SOCS [deg]\",\"type\":\"number\",\"maximum\":720.0,\"minimum\":0.0}},\"requir"
    "ed\":[\"start\",\"stop\",\"increment\"]}}},\"required\":[\"list\"]}},\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"rectangular\":{\"theta_stop\":130.0,\"phi_stop\":270.0,\"program\":{\"laser_"
    "prr\":100000.0,\"name\":\"High Speed\"},\"theta_start\":30.0,\"phi_start\":45.0"
    ",\"theta_increment\":0.04,\"phi_increment\":0.04}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about laser shot files\",\"type\":\"object\",\"prope"
    "rties\":{\"shot_file\":{\"type\":\"object\",\"properties\":{\"file_extension\":{\""
    "description\":\"Shot file extension, without the leading "
    "dot\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"}},\"required\":[\"file_extension\"]}},\"title\":\"Shot"
    " Information\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_extension\":\"sodx\",\"file_uuid\":\"26a00815-67c0-4bff-"
    "8fe8-c577378fe663\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Point filters applied in addition to the "
    "application-defined filters\",\"type\":\"object\",\"title\":\"Stored "
    "filters\",\"properties\":{\"filters\":{\"description\":\"List of point "
    "filters\",\"type\":\"array\",\"items\":{\"description\":\"Point filter definitio"
    "n\",\"type\":\"object\",\"properties\":{\"description\":{\"description\":\"A brief"
    " description of the filter (e.g. for display in a "
    "tooltip)\",\"type\":\"string\"},\"title\":{\"description\":\"A short filter "
    "title (e.g. for display in a selection "
    "list)\",\"type\":\"string\"},\"activated\":{\"description\":\"Apply ('true') or "
    "ignore ('false') this "
    "filter\",\"type\":\"boolean\"},\"filter\":{\"description\":\"The RDB filter "
    "string to apply (e.g. when reading points or index), details see "
    "documentation of function select()\",\"type\":\"string\"}},\"required\":[\"act"
    "ivated\",\"title\",\"description\",\"filter\"]}},\"activated\":{\"description\":\""
    "Apply ('true') or ignore ('false') all filters\",\"type\":\"boolean\"}},\"re"
    "quired\":[\"activated\",\"filters\"],\"$schema\":\"http://json-schema.org/draf"
    "t-04/schema#\"}"
    ,
    /* example: */
    "{\"filters\":[{\"description\":\"Ignore points with uncertain MTA zone "
    "assignment\",\"title\":\"Ignore uncertain points from MTA "
    "resolution\",\"activated\":false,\"filter\":\"riegl.mta_uncertain_point == "
    "0\"},{\"description\":\"Ignore points with an echo signal amplitude below "
    "the dynamic trigger\",\"title\":\"Ignore points below dynamic trigger "
    "level\",\"activated\":false,\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\"},{\"description\":\"Ignore points that are outside the outer AABB in "
    "BOCS\",\"title\":\"Ignore points outside outer "
    "AABB\",\"activated\":false,\"filter\":\"riegl.point_outside_aabb_bocs == "
    "0\"},{\"description\":\"Ignore points that are inside the inner AABB in "
    "BOCS\",\"title\":\"Ignore points inside inner "
    "AABB\",\"activated\":false,\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\"}],\"activated\":true}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details of major system components like lidar sensors,"
    " cameras and navigation devices\",\"type\":\"object\",\"title\":\"System "
    "Description\",\"properties\":{\"$class\":{\"description\":\"Object class name\""
    ",\"type\":\"string\",\"enum\":[\"Document\"]},\"version\":{\"description\":\"Docume"
    "nt format version\",\"type\":\"string\"},\"timestamp\":{\"description\":\"Date "
    "and time of creation (e.g. 2019-06-19T13:32:10+02:00)\",\"type\":\"string\""
    "},\"system\":{\"description\":\"The actual system description, details see "
    "documentation of RIEGL System "
    "Description\",\"type\":\"object\"},\"author\":{\"description\":\"Name of the "
    "author that created the document\",\"type\":\"string\"}},\"required\":[\"$clas"
    "s\",\"version\",\"author\",\"timestamp\",\"system\"],\"$schema\":\"http://json-sch"
    "ema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"$class\":\"Document\",\"version\":\"1.2.1\",\"timestamp\":\"2022-09-30T11:56:2"
    "6+00:00\",\"system\":{\"details\":\"see documentation of RIEGL System "
    "Description\"},\"author\":\"RIEGL LMS GmbH, Austria\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Conversion of background radiation raw values to "
    "temperatures in \\u00b0C\",\"type\":\"object\",\"definitions\":{\"conversion_ta"
    "ble\":{\"type\":\"object\",\"properties\":{\"temperature\":{\"type\":\"array\",\"des"
    "cription\":\"Temperature [\\u00b0C]\",\"items\":{\"type\":\"number\"}},\"value\":{"
    "\"type\":\"array\",\"description\":\"LSB [1]\",\"items\":{\"type\":\"number\"}}},\"re"
    "quired\":[\"value\",\"temperature\"]}},\"title\":\"Temperature Calculation Tab"
    "le\",\"properties\":{\"InGaAs\":{\"$ref\":\"#/definitions/conversion_table\",\"d"
    "escription\":\"Conversion table for InGaAs channel\"},\"Si\":{\"$ref\":\"#/def"
    "initions/conversion_table\",\"description\":\"Conversion table for Si chan"
    "nel\"},\"InGaAs_Si_Difference\":{\"$ref\":\"#/definitions/conversion_table\","
    "\"description\":\"Conversion table for InGaAs - Si "
    "difference\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"InGaAs\":{\"temperature\":[307.22196722535614,309.1153478247277,311.118"
    "8086915047,313.10025350480055,315.2137946389828,317.2172555057597,319."
    "2207163725366,321.2021611858325,323.3157023200148],\"value\":[0.0,64.000"
    "97659230323,128.0019531846065,192.0029297769097,256.0039063692129,320."
    "00488296151616,384.0058595538194,448.0068361461226,512.0078127384259]}"
    ",\"Si\":{\"temperature\":[546.300048828125,548.8164051212026,551.314393850"
    "0972,554.0144257850053,556.604252334815,559.2124464488079,561.80227299"
    "86177,564.4104671126105,567.0002936624203],\"value\":[0.0,64.00097659230"
    "323,128.0019531846065,192.0029297769097,256.0039063692129,320.00488296"
    "151616,384.0058595538194,448.0068361461226,512.0078127384259]},\"InGaAs"
    "_Si_Difference\":{\"temperature\":[1749.977111117893,1749.977111117893,17"
    "49.977111117893,1749.977111117893,1749.977111117893,1749.977111117893,"
    "1744.7813348796044,1681.9971312601092,1622.3944822534868],\"value\":[100"
    "0.0,1100.090029602954,1200.04425183874,1300.1342814416948,1400.0885036"
    "774805,1500.0427259132668,1600.1327555162209,1700.0869777520065,1800.0"
    "411999877924]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Base of timestamps "
    "(epoch)\",\"type\":\"object\",\"title\":\"Time "
    "Base\",\"properties\":{\"epoch\":{\"description\":\"Date and time of timestamp"
    " '0' as proposed by RFC 3339 (e.g. 2015-10-27T00:00:00+01:00).\",\"type\""
    ":\"string\"},\"source\":{\"description\":\"Timestamp source\",\"type\":\"string\","
    "\"enum\":[\"unknown\",\"RTC\",\"GNSS\"]},\"system\":{\"description\":\"Time system "
    "(time standard)\",\"type\":\"string\",\"enum\":[\"unknown\",\"UTC\",\"GPS\"]}},\"req"
    "uired\":[\"epoch\",\"source\"],\"$schema\":\"http://json-schema.org/draft-04/s"
    "chema#\"}"
    ,
    /* example: */
    "{\"epoch\":\"2015-10-27T00:00:00+00:00\",\"source\":\"GNSS\",\"system\":\"UTC\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about position+orientation "
    "files\",\"type\":\"object\",\"title\":\"Trajectory "
    "Information\",\"properties\":{\"company\":{\"description\":\"Company "
    "name\",\"type\":\"string\"},\"location\":{\"description\":\"Project location, "
    "e.g. city/state/country\",\"type\":\"string\"},\"software\":{\"description\":\"S"
    "oftware that calculated the trajectory (this may be the same or "
    "different software than the one that created the "
    "file)\",\"type\":\"string\"},\"time_interval\":{\"description\":\"Time interval "
    "statistics\",\"type\":\"object\",\"properties\":{\"minimum\":{\"description\":\"Mi"
    "nimum time interval "
    "[s]\",\"type\":\"number\"},\"std_dev\":{\"description\":\"Standard deviation of "
    "intervals [s]\",\"type\":\"number\"},\"average\":{\"description\":\"Average time"
    " interval [s]\",\"type\":\"number\"},\"maximum\":{\"description\":\"Maximum time"
    " interval [s]\",\"type\":\"number\"}},\"required\":[\"minimum\",\"average\",\"maxi"
    "mum\",\"std_dev\"]},\"project\":{\"description\":\"Project "
    "name\",\"type\":\"string\"},\"device\":{\"description\":\"Navigation device, "
    "e.g. name/type/serial\",\"type\":\"string\"},\"field_of_application\":{\"descr"
    "iption\":\"Field of application\",\"type\":\"string\",\"enum\":[\"unknown\",\"SLS\""
    ",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"]},\"settings\":{\"description\""
    ":\"Settings used to calculate the trajectory (descriptive "
    "text)\",\"type\":\"string\"},\"navigation_frame\":{\"description\":\"Navigation "
    "frame (NED: North-East-Down, ENU: East-North-Up)\",\"type\":\"string\",\"enu"
    "m\":[\"unknown\",\"NED\",\"ENU\"]}},\"required\":[\"time_interval\",\"navigation_f"
    "rame\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"company\":\"RIEGL LMS\",\"location\":\"Horn\",\"software\":\"Navigation "
    "Software XYZ\",\"time_interval\":{\"minimum\":0.00500032,\"std_dev\":5.51e-07"
    ",\"average\":0.00500053,\"maximum\":0.005004883},\"project\":\"Campaign "
    "3\",\"device\":\"IMU Model 12/1, Serial# 12345\",\"field_of_application\":\"ML"
    "S\",\"settings\":\"default\",\"navigation_frame\":\"NED\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Trigger-Timestamping-Unit/IP "
    "configuration\",\"type\":\"object\",\"title\":\"TTIP "
    "Configuration\",\"properties\":{\"input_usage\":{\"description\":\"0 .. "
    "deactivated, 1 .. just detecting and timestamping\",\"type\":\"integer\",\"m"
    "inimum\":0},\"output_descr\":{\"description\":\"descriptive "
    "string\",\"type\":\"string\"},\"dmi_dist_per_tick\":{\"description\":\"dmi, "
    "distance per tick, just informative "
    "[m]\",\"type\":\"number\"},\"in_polarity\":{\"description\":\"0 .. positive "
    "edge, 1 .. negative edge\",\"type\":\"integer\",\"minimum\":0},\"in_max_delay\""
    ":{\"description\":\"maximum delay to output pulse before fake event is "
    "generated [0.1 msec], zero indicates that no fake events are generated"
    "\",\"type\":\"integer\",\"minimum\":0},\"in_min_duration\":{\"description\":\"inpu"
    "t signals with smaller pulse durations are ignored [0.1 msec]\",\"type\":"
    "\"integer\",\"minimum\":0},\"out_num_pulses\":{\"description\":\"number of "
    "output pulses to be generated, 0 .. infinite\",\"type\":\"integer\",\"minimu"
    "m\":0},\"ext_channel\":{\"description\":\"ID of channel used as external "
    "trigger input, 32 indicates none\",\"type\":\"integer\",\"minimum\":0},\"num_c"
    "hannel\":{\"description\":\"number of input/output channels\",\"type\":\"integ"
    "er\",\"minimum\":0},\"ext_delay\":{\"description\":\"external trigger, time "
    "delay [0.1 msec]\",\"type\":\"integer\",\"minimum\":0},\"ext_subdivider\":{\"des"
    "cription\":\"reduces the frequency, default "
    "1\",\"type\":\"integer\",\"minimum\":0},\"dmi_min_time\":{\"description\":\"dmi, "
    "minimum time interval between trigger events [0.1 msec]\",\"type\":\"integ"
    "er\",\"minimum\":0},\"out_duration\":{\"description\":\"output pulse duration "
    "[0.1 "
    "msec]\",\"type\":\"integer\",\"minimum\":0},\"ext_signal\":{\"description\":\"0 .."
    " use input signal, 1 .. use output signal of channel\",\"type\":\"integer\""
    ",\"minimum\":0},\"in_max_duration\":{\"description\":\"stops measurement of "
    "pulse duration of input signal [0.1 msec]\",\"type\":\"integer\",\"minimum\":"
    "0},\"out_delay\":{\"description\":\"output pulse initial delay after start "
    "[0.1 msec]\",\"type\":\"integer\",\"minimum\":0},\"ttip_version\":{\"description"
    "\":\"following main.sub.ss.sss\",\"type\":\"integer\",\"minimum\":0},\"output_us"
    "age\":{\"description\":\"0 .. no output, 1 .. output derived from internal"
    " clock, 2 .. output derived from dmi, 3 .. output derived from "
    "external signal, 4 .. output static low, 5 .. output static high\",\"typ"
    "e\":\"integer\",\"minimum\":0},\"ext_bitmask\":{\"description\":\"defines which "
    "of the internal pulse generators are to be started\",\"type\":\"integer\",\""
    "minimum\":0},\"out_interval\":{\"description\":\"output pulse interval [0.1 "
    "msec]\",\"type\":\"integer\",\"minimum\":0},\"out_polarity\":{\"description\":\"0 "
    ".. positive edge, 1 .. negative "
    "edge\",\"type\":\"integer\",\"minimum\":0},\"dmi_incr\":{\"description\":\"dmi, "
    "increment in ticks\",\"type\":\"integer\"},\"channel\":{\"description\":\"ID of "
    "input/output channel\",\"type\":\"integer\",\"minimum\":0},\"input_descr\":{\"de"
    "scription\":\"descriptive "
    "string\",\"type\":\"string\"},\"dmi_max_time\":{\"description\":\"dmi, maximum "
    "time interval between trigger events [0.1 msec]\",\"type\":\"integer\",\"min"
    "imum\":0}},\"required\":[\"ttip_version\",\"num_channel\",\"ext_channel\",\"ext_"
    "signal\",\"ext_delay\",\"ext_subdivider\",\"ext_bitmask\",\"dmi_incr\",\"dmi_min"
    "_time\",\"dmi_max_time\",\"dmi_dist_per_tick\",\"channel\",\"output_descr\",\"in"
    "put_descr\",\"output_usage\",\"input_usage\",\"out_polarity\",\"out_duration\","
    "\"out_interval\",\"out_delay\",\"out_num_pulses\",\"in_polarity\",\"in_min_dura"
    "tion\",\"in_max_duration\",\"in_max_delay\"],\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"input_usage\":1,\"output_descr\":\"Port 1 - Trigger\",\"dmi_dist_per_tick\""
    ":0.0,\"in_polarity\":0,\"in_max_delay\":9990,\"in_min_duration\":0,\"out_num_"
    "pulses\":1,\"ext_channel\":32,\"num_channel\":9,\"ext_delay\":0,\"ext_subdivid"
    "er\":1,\"dmi_min_time\":0,\"out_duration\":300,\"ext_signal\":0,\"in_max_durat"
    "ion\":10000,\"out_delay\":0,\"ttip_version\":1,\"output_usage\":1,\"ext_bitmas"
    "k\":0,\"out_interval\":10000,\"out_polarity\":0,\"dmi_incr\":0,\"channel\":0,\"i"
    "nput_descr\":\"Port 1 - Exposure\",\"dmi_max_time\":0}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about vertex file\",\"type\":\"object\",\"properties"
    "\":{\"vertex_file\":{\"type\":\"object\",\"properties\":{\"file_extension\":{\"des"
    "cription\":\"Vertex file extension, without the leading "
    "dot\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\",\"type\":\"string\"}},\"required\":[\"file_exte"
    "nsion\"]}},\"title\":\"Vertex "
    "Information\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_extension\":\"vtx\",\"file_uuid\":\"51534d95-d71f-4f36"
    "-ae1a-3e63a21fd1c7\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the voxels contained in the file\",\"type\""
    ":\"object\",\"definitions\":{\"shape_thresholds\":{\"description\":\"Thresholds"
    " used to compute the voxel's shape_id "
    "value.\",\"type\":\"object\",\"properties\":{\"line\":{\"description\":\"If the "
    "biggest eigenvalue is bigger than the median eigenvalue * "
    "line_threshold, the voxel is considered a line.\",\"type\":\"number\",\"excl"
    "usiveMinimum\":true,\"minimum\":1},\"plane\":{\"description\":\"If the "
    "smallest eigenvalue is smaller than the median eigenvalue * "
    "plane_threshold, the voxel is considered a plane.\",\"type\":\"number\",\"ex"
    "clusiveMinimum\":true,\"maximum\":1,\"minimum\":0,\"exclusiveMaximum\":true}}"
    ",\"required\":[\"plane\",\"line\"]},\"voxel_origin_point\":{\"description\":\"Ori"
    "gin point for all voxel indices in voxel CRS.\",\"type\":\"array\",\"maxItem"
    "s\":3,\"items\":{\"type\":\"number\"},\"minItems\":3},\"voxel_origin_enum\":{\"des"
    "cription\":\"Defines whether the voxel's center or a corner is placed on"
    " CRS origin <tt>(0/0/0)</tt>.\",\"enum\":[\"center\",\"corner\"],\"default\":\"c"
    "orner\"},\"voxel_size\":{\"description\":\"Size of voxels.\",\"type\":\"array\",\""
    "maxItems\":3,\"items\":{\"$ref\":\"#/definitions/edge_length\"},\"minItems\":3}"
    ",\"voxel_size_cubic\":{\"$ref\":\"#/definitions/edge_length\",\"type\":\"number"
    "\"},\"edge_length\":{\"description\":\"Length of voxel edge [m].\",\"type\":\"nu"
    "mber\",\"exclusiveMinimum\":true,\"minimum\":0},\"voxel_type\":{\"description\""
    ":\"Whether a point in a voxel represents its center or its centroid. If"
    " type is <tt>index</tt> there is no point but only an integer voxel in"
    "dex.\",\"enum\":[\"center\",\"centroid\",\"index\"],\"default\":\"centroid\"}},\"tit"
    "le\":\"Voxel Information\",\"oneOf\":[{\"properties\":{\"shape_thresholds\":{\"$"
    "ref\":\"#/definitions/shape_thresholds\"},\"size\":{\"description\":\"Size of "
    "voxels in file coordinate system.\",\"oneOf\":[{\"$ref\":\"#/definitions/vox"
    "el_size\"},{\"$ref\":\"#/definitions/voxel_size_cubic\"}]},\"voxel_origin\":{"
    "\"$ref\":\"#/definitions/voxel_origin_enum\"},\"voxel_type\":{\"$ref\":\"#/defi"
    "nitions/voxel_type\"}},\"required\":[\"size\",\"voxel_origin\",\"voxel_type\"],"
    "\"additionalProperties\":false},{\"properties\":{\"shape_thresholds\":{\"$ref"
    "\":\"#/definitions/shape_thresholds\"},\"size_llcs\":{\"$ref\":\"#/definitions"
    "/voxel_size\",\"description\":\"Size of voxels in a locally levelled "
    "cartesian coordinate system (xyz). This is only used for voxels based "
    "on a map projection.\"},\"reference_point\":{\"description\":\"Point in "
    "WGS84 geodetic decimal degree (EPSG:4326) that was used to compute the"
    " projection distortion parameters. The coefficient order is latitude, "
    "longitude. Only voxels with corresponding geo_tag, voxel_size and "
    "reference_point can be reliably processed together. This entry is "
    "available for voxel files in projected CRS only.\",\"type\":\"array\",\"maxI"
    "tems\":2,\"items\":{\"type\":\"number\",\"minimum\":-180,\"maximum\":180},\"minIte"
    "ms\":2},\"size\":{\"$ref\":\"#/definitions/voxel_size\",\"description\":\"Size "
    "of voxels in file coordinate system.\"},\"voxel_origin\":{\"oneOf\":[{\"$ref"
    "\":\"#/definitions/voxel_origin_enum\"},{\"description\":\"The base point of"
    " the voxel grid. Used together with <tt>voxel_size</tt> and "
    "<tt>voxel_index</tt> to compute actual point coordinates.\",\"$ref\":\"#/d"
    "efinitions/voxel_origin_point\"}]},\"voxel_type\":{\"$ref\":\"#/definitions/"
    "voxel_type\"}},\"required\":[\"reference_point\",\"size_llcs\",\"size\",\"voxel_"
    "origin\",\"voxel_type\"],\"additionalProperties\":false}],\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"shape_thresholds\":{\"plane\":0.16,\"line\":6},\"size_llcs\":[0.51565752528"
    "91171,0.5130835356683303,0.5143705304787237],\"reference_point\":[48,16]"
    ",\"size\":[0.5971642834779395,0.5971642834779395,0.5143705304787237],\"vo"
    "xel_origin\":\"corner\",\"voxel_type\":\"centroid\"}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Settings for waveform "
    "averaging\",\"type\":\"object\",\"title\":\"Waveform Averaging Settings\",\"prop"
    "erties\":{\"num_shots\":{\"type\":\"integer\",\"description\":\"Number of "
    "consecutive shots to be used for averaging.\",\"minimum\":1},\"mta_zone\":{"
    "\"type\":\"integer\",\"description\":\"Fixed MTA zone for averaging.\",\"minimu"
    "m\":1},\"trim\":{\"type\":\"number\",\"description\":\"Percentage for robust ave"
    "raging.\",\"default\":0,\"maximum\":0.5,\"minimum\":0}},\"required\":[\"num_shot"
    "s\",\"mta_zone\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"num_shots\":7,\"mta_zone\":1,\"trim\":0.05}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"description\":\"Details about waveform "
    "files\",\"title\":\"Waveform Information\",\"properties\":{\"sample_data_files"
    "\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"properties\":{\"delta_st\":{\""
    "description\":\"reserved\",\"type\":\"number\"},\"channel_name\":{\"description\""
    ":\"Sample block channel "
    "name\",\"type\":\"string\"},\"laser_wavelength\":{\"description\":\"Laser "
    "wavelength in meters (0 = unknown)\",\"type\":\"number\",\"exclusiveMinimum\""
    ":false,\"minimum\":0},\"file_extension\":{\"description\":\"Sample data file "
    "extension, without the leading dot\",\"type\":\"string\"},\"sample_bits\":{\"t"
    "ype\":\"integer\",\"description\":\"Bitwidth of samples (e.g. 10 bit, 12 bit"
    ")\",\"exclusiveMinimum\":false,\"maximum\":32,\"minimum\":0,\"exclusiveMaximum"
    "\":false},\"channel\":{\"type\":\"integer\",\"description\":\"Sample block "
    "channel number (255 = invalid)\",\"exclusiveMinimum\":false,\"maximum\":255"
    ",\"minimum\":0,\"exclusiveMaximum\":false},\"sample_interval\":{\"description"
    "\":\"Sampling interval in seconds\",\"type\":\"number\",\"exclusiveMinimum\":fa"
    "lse,\"minimum\":0},\"file_uuid\":{\"description\":\"File's Universally Unique"
    " Identifier (RFC 4122)\",\"type\":\"string\"}},\"required\":[\"channel\",\"chann"
    "el_name\",\"sample_interval\",\"sample_bits\",\"laser_wavelength\",\"delta_st\""
    ",\"file_extension\"]}},\"sample_block_file\":{\"type\":\"object\",\"properties\""
    ":{\"file_extension\":{\"description\":\"Sample block file extension, "
    "without the leading "
    "dot\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\",\"type\":\"string\"}},\"required\":[\"file_exte"
    "nsion\"]},\"range_offset_waveform_samples_m\":{\"description\":\"Calibrated "
    "device specific range offset for waveform samples in "
    "meters.\",\"type\":\"number\"},\"range_offset_m\":{\"description\":\"Calibrated "
    "device specific range offset for waveform analysis by system response "
    "fitting in meters.\",\"type\":\"number\"}},\"required\":[\"sample_block_file\","
    "\"sample_data_files\"],\"$schema\":\"http://json-schema.org/draft-04/schema"
    "#\"}"
    ,
    /* example: */
    "{\"sample_data_files\":[{\"delta_st\":0,\"channel_name\":\"high_power\",\"laser"
    "_wavelength\":0,\"file_extension\":\"sp0\",\"sample_bits\":12,\"channel\":0,\"sa"
    "mple_interval\":1.00503e-09,\"file_uuid\":\"da084413-e3e8-4655-a122-071de8"
    "490d8e\"},{\"delta_st\":0,\"channel_name\":\"low_power\",\"laser_wavelength\":0"
    ",\"file_extension\":\"sp1\",\"sample_bits\":12,\"channel\":1,\"sample_interval\""
    ":1.00503e-09,\"file_uuid\":\"93585b5e-5ea9-43a1-947b-e7ba3be642d2\"},{\"del"
    "ta_st\":0,\"channel_name\":\"wwf\",\"laser_wavelength\":0,\"file_extension\":\"s"
    "p5\",\"sample_bits\":12,\"channel\":5,\"sample_interval\":1.00503e-09,\"file_u"
    "uid\":\"9d2298c4-1036-464f-b5cb-1cf8e517f3a0\"}],\"sample_block_file\":{\"fi"
    "le_extension\":\"sbx\",\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\""
    "},\"range_offset_m\":3.1415,\"range_offset_waveform_samples_m \":7.283}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scanner settings for waveform "
    "output\",\"type\":\"array\",\"title\":\"Waveform Settings\",\"items\":{\"type\":\"ob"
    "ject\",\"properties\":{\"pass_ampl_greater\":{\"description\":\"Threshold for "
    "amplitude greater than "
    "[dB]\",\"type\":\"number\"},\"pass_dev_greater\":{\"description\":\"Threshold "
    "for deviation greater than "
    "[1]\",\"type\":\"integer\"},\"enabled\":{\"description\":\"Waveform output "
    "enabled\",\"type\":\"boolean\"},\"sbl_name\":{\"description\":\"Name of sample "
    "block, e.g.: wfm, "
    "wwf\",\"type\":\"string\"},\"smart_enabled\":{\"description\":\"Smart waveform "
    "output "
    "enabled\",\"type\":\"boolean\"},\"logic_expression\":{\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\",\"type\":\"string\"},\"pass_dev_less\":{\"description\":\"Threshold for"
    " deviation less than "
    "[1]\",\"type\":\"integer\"},\"pass_rng_greater\":{\"description\":\"Threshold "
    "for range greater than "
    "[m]\",\"type\":\"number\"},\"channel_idx_mask\":{\"description\":\"Bit mask for "
    "channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = Bit1,"
    " ...\",\"type\":\"integer\"},\"pass_ampl_less\":{\"description\":\"Threshold for"
    " amplitude less than "
    "[dB]\",\"type\":\"number\"},\"pass_rng_less\":{\"description\":\"Threshold for "
    "range less than [m]\",\"type\":\"number\"}},\"required\":[\"sbl_name\",\"enabled"
    "\",\"channel_idx_mask\"]},\"$schema\":\"http://json-schema.org/draft-04/sche"
    "ma#\"}"
    ,
    /* example: */
    "[{\"pass_ampl_greater\":1.0,\"enabled\":true,\"sbl_name\":\"wfm\",\"smart_enabl"
    "ed\":true,\"pass_rng_greater\":9.27,\"channel_idx_mask\":11,\"pass_ampl_less"
    "\":5.0,\"pass_rng_less\":13.11},{\"enabled\":false,\"sbl_name\":\"wwf\",\"channe"
    "l_idx_mask\":32}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Window analysis data estimated from scandata and "
    "resulting filter parameters\",\"type\":\"object\",\"title\":\"Window Analysis\""
    ",\"properties\":{\"settings\":{\"type\":\"object\",\"properties\":{\"amplitude\":{"
    "\"type\":\"object\",\"properties\":{\"additive_value\":{\"type\":\"number\"},\"sigm"
    "a_factor\":{\"type\":\"number\"}},\"required\":[\"sigma_factor\",\"additive_valu"
    "e\"]},\"range\":{\"type\":\"object\",\"properties\":{\"additive_value\":{\"type\":\""
    "number\"},\"sigma_factor\":{\"type\":\"number\"}},\"required\":[\"sigma_factor\","
    "\"additive_value\"]}},\"required\":[\"range\",\"amplitude\"]},\"filter\":{\"type\""
    ":\"object\",\"properties\":{\"range_max\":{\"type\":\"array\",\"description\":\"[m]"
    "\",\"items\":{\"type\":\"number\"}},\"angle\":{\"type\":\"array\",\"description\":\"[d"
    "eg]\",\"items\":{\"type\":\"number\"}},\"amplitude_max\":{\"type\":\"array\",\"descr"
    "iption\":\"[dB]\",\"items\":{\"type\":\"number\"}},\"range_min\":{\"type\":\"array\","
    "\"description\":\"[m]\",\"items\":{\"type\":\"number\"}}},\"required\":[\"angle\",\"r"
    "ange_min\",\"range_max\",\"amplitude_max\"]},\"result\":{\"type\":\"object\",\"pro"
    "perties\":{\"range_sigma\":{\"type\":\"array\",\"description\":\"[m]\",\"items\":{\""
    "type\":\"number\"}},\"amplitude_mean\":{\"type\":\"array\",\"description\":\"[dB]\""
    ",\"items\":{\"type\":\"number\"}},\"range_mean\":{\"type\":\"array\",\"description\""
    ":\"[m]\",\"items\":{\"type\":\"number\"}},\"timestamp\":{\"type\":\"array\",\"descrip"
    "tion\":\"[s]\",\"items\":{\"type\":\"number\"}},\"amplitude_offset\":{\"type\":\"arr"
    "ay\",\"description\":\"[dB]\",\"items\":{\"type\":\"number\"}},\"angle\":{\"type\":\"a"
    "rray\",\"description\":\"[deg]\",\"items\":{\"type\":\"number\"}},\"amplitude_sigm"
    "a\":{\"type\":\"array\",\"description\":\"[dB]\",\"items\":{\"type\":\"number\"}}},\"r"
    "equired\":[\"angle\",\"range_mean\",\"range_sigma\",\"amplitude_mean\",\"amplitu"
    "de_sigma\",\"amplitude_offset\"]}},\"required\":[\"result\",\"filter\",\"setting"
    "s\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"settings\":{\"amplitude\":{\"additive_value\":1.0,\"sigma_factor\":4},\"rang"
    "e\":{\"additive_value\":0.1,\"sigma_factor\":8}},\"filter\":{\"range_max\":[0.4"
    "24,0.425,0.426,0.427,0.428,0.428,0.429,0.43,0.431,0.431,0.432],\"angle\""
    ":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],\"amplitude_m"
    "ax\":[8.04,8.01,7.99,7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78],\"range_min\""
    ":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.218,-0.219,-0.221,-0.223,-0.225"
    ",-0.227]},\"result\":{\"range_sigma\":[0.01869652,0.02151435,0.01747969,0."
    "01918765,0.01945776,0.01934862,0.01955329,0.02225589,0.02229977,0.0189"
    "9122,0.02009433],\"amplitude_mean\":[5.347396,5.263155,5.224655,5.179926"
    ",5.097782,5.116479,5.051756,4.983473,5.007885,5.002441,4.982],\"range_m"
    "ean\":[0.1105621,0.1079564,0.1087088,0.1067261,0.1054582,0.1090412,0.10"
    "2871,0.1019044,0.1051523,0.1058445,0.1031261],\"timestamp\":[408.4441,41"
    "1.4443],\"amplitude_offset\":[1.9,1.9],\"angle\":[14.9,15.0,15.1,15.2,15.3"
    ",15.4,15.5,15.6,15.7,15.8,15.9],\"amplitude_sigma\":[0.4272844,0.4298479"
    ",0.4236816,0.4283583,0.4362353,0.4315141,0.4373984,0.4472798,0.4346001"
    ",0.4345487,0.4540681]}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Correction parameters for window glass "
    "echoes\",\"type\":\"object\",\"title\":\"Window Echo Correction\",\"properties\":"
    "{\"amplitude\":{\"type\":\"object\",\"description\":\"Amplitude axis of "
    "correction "
    "table\",\"properties\":{\"minimum\":{\"type\":\"number\",\"description\":\"Minimum"
    " amplitude in "
    "dB\",\"minimum\":0.0},\"entries\":{\"type\":\"integer\",\"description\":\"Number "
    "of amplitude entries\",\"minimum\":1},\"maximum\":{\"type\":\"number\",\"descrip"
    "tion\":\"Maximum amplitude in dB\",\"minimum\":0.0}},\"required\":[\"minimum\","
    "\"maximum\",\"entries\"]},\"slices\":{\"type\":\"array\",\"items\":{\"type\":\"object"
    "\",\"description\":\"Window echo correction parameter slice\",\"properties\":"
    "{\"amplitude\":{\"type\":\"number\",\"description\":\"Window echo amplitude of "
    "slice in dB\"},\"table\":{\"type\":\"array\",\"description\":\"Correction table "
    "(dimension defined by the 'amplitude' and 'range' "
    "objects)\",\"items\":{\"type\":\"array\",\"description\":\"Table row (= "
    "amplitude axis)\",\"items\":{\"type\":\"array\",\"description\":\"Table column "
    "(= range axis)\",\"maxItems\":3,\"items\":{\"description\":\"Table cell (item "
    "0: amplitude in dB, 1: range in m, 2: flags)\",\"type\":\"number\"},\"minIte"
    "ms\":3},\"minItems\":1},\"minItems\":1}},\"required\":[\"amplitude\",\"table\"]}}"
    ",\"range\":{\"type\":\"object\",\"description\":\"Range axis of correction "
    "table\",\"properties\":{\"minimum\":{\"type\":\"number\",\"description\":\"Minimum"
    " range in m\",\"maximum\":2.0,\"minimum\":-2.0},\"entries\":{\"type\":\"integer\""
    ",\"description\":\"Number of range entries\",\"minimum\":1},\"maximum\":{\"type"
    "\":\"number\",\"description\":\"Maximum range in m\",\"maximum\":2.0,\"minimum\":"
    "-2.0}},\"required\":[\"minimum\",\"maximum\",\"entries\"]}},\"required\":[\"ampli"
    "tude\",\"range\",\"slices\"],\"$schema\":\"http://json-schema.org/draft-04/sch"
    "ema#\"}"
    ,
    /* example: */
    "{\"amplitude\":{\"minimum\":2,\"entries\":128,\"maximum\":20},\"slices\":[{\"ampl"
    "itude\":1.5,\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]},{\"amplitude"
    "\":2.0,\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]}],\"range\":{\"minim"
    "um\":-1.5060822940732335,\"entries\":128,\"maximum\":1.5060822940732335}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const std::size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);
const std::vector<const MetaDataItemInfo*> RDB_META_DATA_ITEMS_VECTOR(
    RDB_META_DATA_ITEMS, RDB_META_DATA_ITEMS + RDB_META_DATA_ITEM_COUNT
);

}}} // namespace riegl::rdb::pointcloud

#endif // RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
