/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 *
 *******************************************************************************
 */

#ifndef RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
#define RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87

#include <vector>
#include <string>
#include <cstdlib>
#include <cstdint>

namespace riegl {
namespace rdb {
namespace pointcloud {

// Meta data item definition version
const std::string RDB_META_DATA_ITEMS_VERSION = "1.1.56";
const std::string RDB_META_DATA_ITEMS_DATE = "2025-07-01";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
struct MetaDataItemInfo
{
    std::string   name;
    std::string   title;
    std::string   description;
    unsigned char status;
    std::string   schema;
    std::string   example;

    operator const std::string&() const { return name; }
};

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Angular Notch "
    "Filter\",\"properties\":{\"range_mean\":{\"description\":\"Mean range [m]\",\"it"
    "ems\":{\"type\":\"number\"},\"type\":\"array\"},\"angle\":{\"description\":\"Angle ["
    "deg]\",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"amplitude_mean\":{\"des"
    "cription\":\"Mean amplitude [dB]\",\"items\":{\"type\":\"number\"},\"type\":\"arra"
    "y\"}},\"description\":\"Angular notch filter parameters for window glass e"
    "choes\",\"required\":[\"angle\",\"range_mean\",\"amplitude_mean\"],\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"range_mean\":[0.094,0.094,0.09075,0.08675,0.0815,0.0775,0.074,0.071,0"
    ".068,0.0675,0.06475],\"angle\":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,"
    "22.0,23.0,24.0],\"amplitude_mean\":[3.780913,3.780913,3.480913,3.120913,"
    "2.850913,2.720913,2.680913,2.610913,2.530913,2.570913,2.570913]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Atmospheric "
    "Parameters\",\"properties\":{\"pressure\":{\"description\":\"Pressure along "
    "measurment path "
    "[mbar]\",\"type\":\"number\"},\"wavelength\":{\"description\":\"Laser wavelength"
    " [nm]\",\"type\":\"number\"},\"temperature\":{\"description\":\"Temperature "
    "along measurement path "
    "[\\u00b0C]\",\"type\":\"number\"},\"group_velocity\":{\"description\":\"Group "
    "velocity of laser beam "
    "[m/s]\",\"type\":\"number\"},\"amsl\":{\"description\":\"Height above mean sea "
    "level (AMSL) "
    "[m]\",\"type\":\"number\"},\"pressure_sl\":{\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\",\"type\":\"number\"},\"attenuation\":{\"description\":\"Atmospheric "
    "attenuation "
    "[1/km]\",\"type\":\"number\"},\"rel_humidity\":{\"description\":\"Relative "
    "humidity along measurement path "
    "[%]\",\"type\":\"number\"}},\"description\":\"Atmospheric parameters\",\"require"
    "d\":[\"temperature\",\"pressure\",\"rel_humidity\",\"pressure_sl\",\"amsl\",\"grou"
    "p_velocity\",\"attenuation\",\"wavelength\"],\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"pressure\":970,\"wavelength\":1550,\"temperature\":7,\"group_velocity\":299"
    "711000.0,\"amsl\":0,\"pressure_sl\":970,\"attenuation\":0.028125,\"rel_humidi"
    "ty\":63}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Beam "
    "Geometry\",\"properties\":{\"beam_exit_diameter\":{\"description\":\"Beam "
    "width at exit aperture [m]\",\"exclusiveMinimum\":false,\"type\":\"number\",\""
    "minimum\":0},\"beam_divergence\":{\"description\":\"Beam divergence in far "
    "field [rad]\",\"exclusiveMinimum\":false,\"type\":\"number\",\"minimum\":0}},\"d"
    "escription\":\"Laser beam geometry details\",\"required\":[\"beam_exit_diame"
    "ter\",\"beam_divergence\"],\"$schema\":\"http://json-schema.org/draft-04/sch"
    "ema#\"}"
    ,
    /* example: */
    "{\"beam_exit_diameter\":0.0072,\"beam_divergence\":0.0003}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Control Object Catalog\",\"properties\":{\"types"
    "\":{\"items\":{\"type\":\"object\",\"oneOf\":[{\"$ref\":\"#/definitions/rectangle\""
    "},{\"$ref\":\"#/definitions/checkerboard2x2\"},{\"$ref\":\"#/definitions/chev"
    "ron\"},{\"$ref\":\"#/definitions/circular_disk\"},{\"$ref\":\"#/definitions/cy"
    "linder\"},{\"$ref\":\"#/definitions/sphere\"},{\"$ref\":\"#/definitions/round_"
    "corner_cube_prism\"}]},\"type\":\"array\",\"uniqueItems\":true}},\"description"
    "\":\"List of control object type definitions\",\"definitions\":{\"rectangle\""
    ":{\"description\":\"rectangle\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/defini"
    "tions/common\"},{\"description\":\"rectangle specific properties\",\"type\":\""
    "object\",\"required\":[\"shape\",\"length\",\"width\"],\"properties\":{\"width\":{\""
    "description\":\"width in meters\",\"exclusiveMinimum\":true,\"type\":\"number\""
    ",\"minimum\":0},\"shape\":{\"type\":\"string\",\"enum\":[\"rectangle\"]},\"length\":"
    "{\"description\":\"length in meters\",\"exclusiveMinimum\":true,\"type\":\"numb"
    "er\",\"minimum\":0}}}]},\"circular_disk\":{\"description\":\"circular disk\",\"t"
    "ype\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"description\":"
    "\"circular disk specific properties\",\"type\":\"object\",\"required\":[\"shape"
    "\",\"diameter\"],\"properties\":{\"diameter\":{\"description\":\"diameter in met"
    "ers\",\"exclusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},\"offset\":{\"d"
    "escription\":\"offset in meters, e.g. reflector constant "
    "(optional)\",\"type\":\"number\"},\"shape\":{\"description\":\"shape identifier\""
    ",\"type\":\"string\",\"enum\":[\"circular_disk\"]}}}]},\"round_corner_cube_pris"
    "m\":{\"description\":\"round corner cube prism\",\"type\":\"object\",\"allOf\":[{"
    "\"$ref\":\"#/definitions/common\"},{\"description\":\"round corner cube prism"
    " specific properties\",\"type\":\"object\",\"required\":[\"shape\",\"diameter\"],"
    "\"properties\":{\"diameter\":{\"description\":\"diameter in meters\",\"exclusiv"
    "eMinimum\":true,\"type\":\"number\",\"minimum\":0},\"offset\":{\"description\":\"o"
    "ffset in meters, e.g. reflector constant "
    "(optional)\",\"type\":\"number\"},\"shape\":{\"description\":\"shape identifier\""
    ",\"type\":\"string\",\"enum\":[\"round_corner_cube_prism\"]}}}]},\"checkerboard"
    "2x2\":{\"description\":\"checkerboard 2 by 2\",\"type\":\"object\",\"allOf\":[{\"$"
    "ref\":\"#/definitions/common\"},{\"description\":\"checkerboard specific pro"
    "perties\",\"type\":\"object\",\"required\":[\"shape\",\"square_length\"],\"propert"
    "ies\":{\"square_length\":{\"description\":\"length of a square of the "
    "checkerboard in meters\",\"exclusiveMinimum\":true,\"type\":\"number\",\"minim"
    "um\":0},\"shape\":{\"description\":\"shape identifier\",\"type\":\"string\",\"enum"
    "\":[\"checkerboard2x2\"]}}}]},\"common\":{\"description\":\"common object prop"
    "erties\",\"type\":\"object\",\"required\":[\"name\",\"shape\"],\"properties\":{\"des"
    "cription\":{\"description\":\"string describing the "
    "object\",\"type\":\"string\"},\"name\":{\"description\":\"unique type identifier"
    "\",\"minLength\":3,\"type\":\"string\"},\"shape\":{\"description\":\"shape identif"
    "ier\",\"type\":\"string\",\"enum\":[\"rectangle\",\"checkerboard2x2\",\"chevron\",\""
    "circular_disk\",\"cylinder\",\"sphere\",\"round_corner_cube_prism\"]},\"surfac"
    "e_type\":{\"description\":\"surface material type\",\"type\":\"string\",\"enum\":"
    "[\"retro_reflective_foil\",\"diffuse\"]}}},\"sphere\":{\"description\":\"sphere"
    "\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"descripti"
    "on\":\"sphere specific properties\",\"type\":\"object\",\"required\":[\"shape\",\""
    "diameter\"],\"properties\":{\"diameter\":{\"description\":\"diameter in meters"
    "\",\"exclusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},\"shape\":{\"descr"
    "iption\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"sphere\"]}}}]},\"cyl"
    "inder\":{\"description\":\"cylinder\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/d"
    "efinitions/common\"},{\"description\":\"cylinder specific properties\",\"typ"
    "e\":\"object\",\"required\":[\"shape\",\"diameter\",\"height\"],\"properties\":{\"he"
    "ight\":{\"description\":\"height in meters\",\"exclusiveMinimum\":true,\"type\""
    ":\"number\",\"minimum\":0},\"diameter\":{\"description\":\"diameter in meters\","
    "\"exclusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},\"shape\":{\"descrip"
    "tion\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"cylinder\"]}}}]},\"che"
    "vron\":{\"description\":\"chevron\",\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/def"
    "initions/common\"},{\"description\":\"chevron specific properties\",\"type\":"
    "\"object\",\"required\":[\"shape\",\"outside_edge_length\",\"thickness\"],\"prope"
    "rties\":{\"outside_edge_length\":{\"description\":\"length of the two outer "
    "edges in meters\",\"exclusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},"
    "\"thickness\":{\"description\":\"thickness in meters\",\"exclusiveMinimum\":tr"
    "ue,\"type\":\"number\",\"minimum\":0},\"shape\":{\"description\":\"shape identifi"
    "er\",\"type\":\"string\",\"enum\":[\"chevron\"]}}}]}},\"required\":[\"types\"],\"$sc"
    "hema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or "
    "'diffuse')\"],\"types\":[{\"description\":\"Rectangle (60cm x "
    "30cm)\",\"width\":0.3,\"name\":\"Rectangle "
    "60x30\",\"shape\":\"rectangle\",\"length\":0.6},{\"description\":\"Rectangle "
    "(80cm x 40cm)\",\"width\":0.4,\"name\":\"Rectangle "
    "80x40\",\"shape\":\"rectangle\",\"length\":0.8},{\"description\":\"Checkerboard "
    "(square length: 30cm)\",\"square_length\":0.3,\"name\":\"Checkerboard2x2 "
    "30\",\"shape\":\"checkerboard2x2\"},{\"description\":\"Checkerboard (square "
    "length: 50cm)\",\"square_length\":0.5,\"name\":\"Checkerboard2x2 "
    "50\",\"shape\":\"checkerboard2x2\"},{\"description\":\"Chevron (a=24''; b=4'')"
    "\",\"outside_edge_length\":0.6096,\"thickness\":0.1016,\"name\":\"Chevron "
    "24''/4''\",\"shape\":\"chevron\"},{\"description\":\" Circular Disk (diameter:"
    " 50cm)\",\"diameter\":0.5,\"name\":\"Circular Disk 50\",\"shape\":\"circular_dis"
    "k\",\"surface_type\":\"diffuse\"},{\"diameter\":0.05,\"offset\":0.0,\"name\":\"RIE"
    "GL flat reflector 50 mm\",\"shape\":\"circular_disk\",\"description\":\"flat "
    "circular reflector from retro-reflective foil\",\"surface_type\":\"retro_r"
    "eflective_foil\"},{\"diameter\":0.1,\"offset\":0.0,\"name\":\"RIEGL flat "
    "reflector 100 mm\",\"shape\":\"circular_disk\",\"description\":\"flat circular"
    " reflector from retro-reflective foil\",\"surface_type\":\"retro_reflectiv"
    "e_foil\"},{\"diameter\":0.15,\"offset\":0.0,\"name\":\"RIEGL flat reflector "
    "150 mm\",\"shape\":\"circular_disk\",\"description\":\"flat circular reflector"
    " from retro-reflective foil\",\"surface_type\":\"retro_reflective_foil\"},{"
    "\"diameter\":0.05,\"height\":0.05,\"shape\":\"cylinder\",\"description\":\"cylind"
    "rical reflector from retro-reflective foil\",\"name\":\"RIEGL cylindrical "
    "reflector 50 mm\",\"surface_type\":\"retro_reflective_foil\"},{\"diameter\":0"
    ".1,\"height\":0.1,\"shape\":\"cylinder\",\"description\":\"cylindrical "
    "reflector from retro-reflective foil\",\"name\":\"RIEGL cylindrical "
    "reflector 100 "
    "mm\",\"surface_type\":\"retro_reflective_foil\"},{\"description\":\"Sphere "
    "(diameter: 200 mm)\",\"diameter\":0.2,\"name\":\"Sphere 200 "
    "mm\",\"shape\":\"sphere\"},{\"description\":\"round corner cube "
    "prism\",\"diameter\":0.05,\"offset\":0.0,\"name\":\"Corner Cube Prism 50 "
    "mm\",\"shape\":\"round_corner_cube_prism\"}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the control object reference "
    "file\",\"type\":\"object\",\"title\":\"Control Object Reference "
    "file\",\"properties\":{\"reference_file\":{\"description\":\"Reference to a "
    "control object file\",\"type\":\"object\",\"required\":[\"file_uuid\",\"file_pat"
    "h\"],\"properties\":{\"file_path\":{\"description\":\"Path of the control "
    "object file relative to referring "
    "file\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"Control object "
    "file's Universally Unique Identifier (RFC 4122)\",\"type\":\"string\"}}}},\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoi"
    "nts.cpx\",\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Detection "
    "Probability\",\"properties\":{\"amplitude\":{\"description\":\"Amplitude [dB]\""
    ",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"detection_probability\":{\"d"
    "escription\":\"Detection probability [0..1]\",\"items\":{\"type\":\"number\"},\""
    "type\":\"array\"}},\"description\":\"Detection probability as a function of "
    "amplitude\",\"required\":[\"amplitude\",\"detection_probability\"],\"$schema\":"
    "\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],\"detection"
    "_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device "
    "Information\",\"properties\":{\"device_name\":{\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\",\"type\":\"string\"},\"channel_number\":{\"description\":\"Laser "
    "channel number (not defined or 0: single channel device)\",\"exclusiveMi"
    "nimum\":false,\"type\":\"integer\",\"minimum\":0},\"channel_text\":{\"descriptio"
    "n\":\"Optional channel description (e.g. 'Green Channel' for "
    "multi-channel "
    "devices)\",\"type\":\"string\"},\"device_build\":{\"description\":\"Device build"
    " variant\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"serial_number\":{\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\",\"type\":\"string\"}},\"description\":\"Details about the device "
    "used to acquire the point cloud\",\"required\":[\"device_type\",\"serial_num"
    "ber\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"device_name\":\"Scanner 1\",\"channel_number\":0,\"channel_text\":\"\",\"devic"
    "e_build\":\"\",\"device_type\":\"VZ-400i\",\"serial_number\":\"S2221234\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device "
    "Geometry\",\"properties\":{\"amu\":{\"description\":\"Angle Measurement "
    "Unit\",\"type\":\"object\",\"properties\":{\"frameCC\":{\"description\":\"Frame "
    "Circle Count (number of LSBs per full rotation about frame axis)\",\"exc"
    "lusiveMinimum\":false,\"type\":\"number\",\"minimum\":0},\"lineCC\":{\"descripti"
    "on\":\"Line Circle Count (number of LSBs per full rotation about line ax"
    "is)\",\"exclusiveMinimum\":false,\"type\":\"number\",\"minimum\":0}}},\"primary\""
    ":{\"description\":\"Primary device geometry structure (mandatory)\",\"type\""
    ":\"object\",\"required\":[\"ID\",\"content\"],\"properties\":{\"content\":{\"descri"
    "ption\":\"Internal calibration values\",\"items\":{\"type\":\"number\"},\"type\":"
    "\"array\"},\"ID\":{\"description\":\"Structure identifier\",\"items\":{\"type\":\"i"
    "nteger\"},\"type\":\"array\",\"maxItems\":2,\"minItems\":2}}},\"secondary\":{\"des"
    "cription\":\"Additional device geometry structure (optional)\",\"type\":\"ob"
    "ject\",\"required\":[\"ID\",\"content\"],\"properties\":{\"content\":{\"descriptio"
    "n\":\"Internal calibration values\",\"items\":{\"type\":\"number\"},\"type\":\"arr"
    "ay\"},\"ID\":{\"description\":\"Structure identifier\",\"items\":{\"type\":\"integ"
    "er\"},\"type\":\"array\",\"maxItems\":2,\"minItems\":2}}}},\"description\":\"Scann"
    "er device geometry details\",\"required\":[\"primary\"],\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amu\":{\"frameCC\":124000,\"lineCC\":124000},\"primary\":{\"content\":[0],\"ID"
    "\":[4,0]},\"secondary\":{\"content\":[0],\"ID\":[91,0]}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Geometry Passive "
    "Channel\",\"properties\":{\"primary\":{\"description\":\"Primary device "
    "geometry structure (mandatory)\",\"type\":\"object\",\"required\":[\"ID\",\"cont"
    "ent\"],\"properties\":{\"content\":{\"description\":\"Internal calibration val"
    "ues\",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"ID\":{\"description\":\"St"
    "ructure identifier\",\"items\":{\"type\":\"integer\"},\"type\":\"array\",\"maxItem"
    "s\":2,\"minItems\":2}}}},\"description\":\"Scanner passive channel device "
    "geometry details\",\"required\":[\"primary\"],\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"primary\":{\"content\":[0],\"ID\":[143,0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Limits of the measured values output by the device. "
    "The limits depend on the device type, measurement program and/or scan "
    "pattern.\",\"type\":\"object\",\"title\":\"Device Output "
    "Limits\",\"properties\":{\"deviation_minimum\":{\"description\":\"Minimum "
    "possible pulse shape deviation.\",\"type\":\"number\"},\"background_radiatio"
    "n_minimum\":{\"description\":\"Minimum possible background radiation.\",\"ty"
    "pe\":\"number\"},\"reflectance_minimum\":{\"description\":\"Minimum possible "
    "reflectance in "
    "dB.\",\"type\":\"number\"},\"range_minimum\":{\"description\":\"Minimum possible"
    " range in "
    "meters.\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum "
    "possible range in "
    "meters.\",\"type\":\"number\"},\"echo_count_maximum\":{\"description\":\"Maximum"
    " number of echoes a laser shot can "
    "have.\",\"type\":\"number\"},\"reflectance_maximum\":{\"description\":\"Maximum "
    "possible reflectance in "
    "dB.\",\"type\":\"number\"},\"mta_zone_count_maximum\":{\"description\":\"Maximum"
    " number of MTA zones.\",\"type\":\"number\"},\"background_radiation_maximum\""
    ":{\"description\":\"Maximum possible background radiation.\",\"type\":\"numbe"
    "r\"},\"amplitude_minimum\":{\"description\":\"Minimum possible amplitude in "
    "dB.\",\"type\":\"number\"},\"deviation_maximum\":{\"description\":\"Maximum "
    "possible pulse shape deviation.\",\"type\":\"number\"},\"amplitude_maximum\":"
    "{\"description\":\"Maximum possible amplitude in dB.\",\"type\":\"number\"}},\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"mta_zone_count_maximum\":7,\"background_radiation_maximum\":0,\"reflecta"
    "nce_maximum\":100.0,\"deviation_minimum\":-1,\"background_radiation_minimu"
    "m\":0,\"deviation_maximum\":32767,\"reflectance_minimum\":-100.0,\"amplitude"
    "_minimum\":0.0,\"range_minimum\":2.9,\"amplitude_maximum\":100.0,\"range_max"
    "imum\":10000.0}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the devices used to acquire the point "
    "cloud (e.g. when merging point clouds of different devices)\",\"items\":{"
    "\"type\":\"object\",\"required\":[\"device_type\",\"serial_number\"],\"properties"
    "\":{\"device_name\":{\"description\":\"Optional device name (e.g. 'Scanner "
    "1' for multi-scanner "
    "systems)\",\"type\":\"string\"},\"signed\":{\"description\":\"Flag that is set "
    "when the original 'riegl.device' entry in the source file was "
    "correctly "
    "signed.\",\"type\":\"boolean\"},\"channel_number\":{\"description\":\"Laser "
    "channel number (not defined or 0: single channel device)\",\"exclusiveMi"
    "nimum\":false,\"type\":\"integer\",\"minimum\":0},\"channel_text\":{\"descriptio"
    "n\":\"Optional channel description (e.g. 'Green Channel' for "
    "multi-channel "
    "devices)\",\"type\":\"string\"},\"device_build\":{\"description\":\"Device build"
    " variant\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"serial_number\":{\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\",\"type\":\"string\"}}},\"type\":\"array\",\"title\":\"Devices "
    "Information\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "[{\"device_name\":\"Scanner 1\",\"signed\":false,\"channel_number\":0,\"channel"
    "_text\":\"\",\"device_build\":\"\",\"device_type\":\"VZ-400i\",\"serial_number\":\"S"
    "2221234\"},{\"device_name\":\"Scanner 2\",\"signed\":true,\"channel_number\":1,"
    "\"channel_text\":\"\",\"device_build\":\"\",\"device_type\":\"VQ-1560i-DW\",\"seria"
    "l_number\":\"S2222680\"},{\"device_name\":\"Scanner 3\",\"signed\":true,\"channe"
    "l_number\":2,\"channel_text\":\"\",\"device_build\":\"\",\"device_type\":\"VQ-1560"
    "i-DW\",\"serial_number\":\"S2222680\"}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"DMI Configuration\",\"properties\":{\"dmi_max_ti"
    "me\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"Maximum time interval"
    " between trigger events, 0 means timeout is inactive [0.1 msec]\",\"excl"
    "usiveMinimum\":false,\"exclusiveMaximum\":false,\"maximum\":65535},\"dmi_cou"
    "nt_dir\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"DMI count "
    "direction is inverted\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":fal"
    "se,\"maximum\":1},\"dmi_dist_per_tick\":{\"description\":\"Distance per tick,"
    " just informative [m]\",\"type\":\"number\"},\"dmi_incr\":{\"type\":\"integer\",\""
    "minimum\":0,\"description\":\"Increment in ticks, 0 means distance trigger"
    " is deactivated\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":false,\"ma"
    "ximum\":2147483647},\"dmi_min_time\":{\"type\":\"integer\",\"minimum\":0,\"descr"
    "iption\":\"Minimum time interval between trigger events, 0 means rate "
    "limitation is inactive [0.1 msec]\",\"exclusiveMinimum\":false,\"exclusive"
    "Maximum\":false,\"maximum\":65535}},\"description\":\"Distance measuring "
    "instrument configuration\",\"required\":[\"dmi_count_dir\",\"dmi_incr\",\"dmi_"
    "min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\"],\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"dmi_max_time\":100,\"dmi_count_dir\":0,\"dmi_dist_per_tick\":0.0002439999"
    "9529123306,\"dmi_incr\":0,\"dmi_min_time\":0}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Dyntrig\",\"title\":\"Dyntrig\",\"properties\":{\"scale\":{\"des"
    "cription\":\"Scale in units of dB, for calculation of line: line number "
    "N=(amplitude-offset)/scale\",\"type\":\"number\"},\"offset\":{\"description\":\""
    "Offset in units of dB, for calculation of "
    "line\",\"type\":\"number\"},\"post\":{\"description\":\"Dyntrig post values\",\"ty"
    "pe\":\"object\",\"required\":[\"nc\",\"a\"],\"properties\":{\"nc\":{\"description\":\""
    "Number of entries\",\"type\":\"integer\",\"maximum\":255,\"minimum\":0},\"a\":{\"i"
    "tems\":{\"type\":\"number\"},\"description\":\"...\",\"type\":\"array\",\"maxItems\":"
    "2080,\"minItems\":1}}},\"tab\":{\"items\":{\"description\":\"...\",\"type\":\"objec"
    "t\",\"required\":[\"thra1\",\"del1\",\"dec1\",\"thra2\",\"del2\",\"dec2\"],\"propertie"
    "s\":{\"dec2\":{\"description\":\"Rise of pre-trigger threshold after del2 in"
    " units of dB/m\",\"type\":\"number\"},\"dec1\":{\"description\":\"Decay of "
    "post-trigger threshold after del1 in units of "
    "dB/m\",\"type\":\"number\"},\"thra1\":{\"description\":\"Post-trigger threshold "
    "in units of dB\",\"type\":\"number\"},\"del1\":{\"description\":\"Length of "
    "constant post-trigger threshold elevation in units of "
    "m\",\"type\":\"number\"},\"del2\":{\"description\":\"Length of constant "
    "pre-trigger threshold elevation in units of "
    "m\",\"type\":\"number\"},\"thra2\":{\"description\":\"Pre-trigger threshold in "
    "units of dB\",\"type\":\"number\"}}},\"description\":\"...\",\"type\":\"array\",\"ma"
    "xItems\":128,\"minItems\":1},\"pre\":{\"description\":\"Dyntrig pre values\",\"t"
    "ype\":\"object\",\"required\":[\"nc\",\"a\"],\"properties\":{\"nc\":{\"description\":"
    "\"Number of entries\",\"type\":\"integer\",\"maximum\":255,\"minimum\":0},\"a\":{\""
    "items\":{\"type\":\"number\"},\"description\":\"...\",\"type\":\"array\",\"maxItems\""
    ":2080,\"minItems\":1}}}},\"$schema\":\"http://json-schema.org/draft-04/sche"
    "ma#\"}"
    ,
    /* example: */
    "{\"scale\":1.0,\"offset\":0.0,\"post\":{\"nc\":128,\"a\":[78,86,126,134,31]},\"ta"
    "b\":[{\"dec2\":0.0,\"dec1\":0.0,\"thra1\":0.0,\"del1\":0.0,\"del2\":0.0,\"thra2\":0"
    ".0}],\"pre\":{\"nc\":128,\"a\":[78,86,126,134,31]}}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Echo Information\",\"properties\":{\"echo_file\":"
    "{\"type\":\"object\",\"required\":[\"file_extension\"],\"properties\":{\"file_uui"
    "d\":{\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Echo file "
    "extension, without the leading "
    "dot\",\"type\":\"string\"}}}},\"description\":\"Details about echo files\",\"req"
    "uired\":[\"echo_file\"],\"$schema\":\"http://json-schema.org/draft-04/schema"
    "#\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_uuid\":\"26a03615-67c0-4bea-8fe8-c577378fe661\",\"file"
    "_extension\":\"owp\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Exponential "
    "Decomposition\",\"description\":\"Details for exponential decomposition of"
    " full waveform "
    "data\",\"patternProperties\":{\"^[0-9]+$\":{\"description\":\"one field per "
    "channel, field name is channel index\",\"$ref\":\"#/definitions/channel\"}}"
    ",\"definitions\":{\"channel\":{\"type\":\"object\",\"required\":[\"delay\",\"scale\""
    ",\"parameter\"],\"properties\":{\"scale\":{\"description\":\"amplitude "
    "calibration\",\"type\":\"number\"},\"parameter\":{\"description\":\"parameters "
    "of the syswave exponential sum\",\"type\":\"object\",\"required\":[\"A\",\"B\",\"g"
    "amma\",\"omega\"],\"properties\":{\"gamma\":{\"description\":\"decay in 1/second"
    "\",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"B\":{\"description\":\"imagin"
    "ary part of amplitude factor in units of full-scale\",\"items\":{\"type\":\""
    "number\"},\"type\":\"array\"},\"omega\":{\"description\":\"angular frequency in "
    "Hz\",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"A\":{\"description\":\"real"
    " part of amplitude factor in units of full-scale\",\"items\":{\"type\":\"num"
    "ber\"},\"type\":\"array\"}}},\"delay\":{\"description\":\"delay calibration in s"
    "econds\",\"type\":\"number\"},\"a_lin\":{\"type\":\"number\",\"minimum\":0,\"descrip"
    "tion\":\"relative linear amplitude range [0..1]\",\"exclusiveMinimum\":fals"
    "e,\"exclusiveMaximum\":false,\"maximum\":1}}}},\"additionalProperties\":fals"
    "e,\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"1\":{\"scale\":1.0,\"parameter\":{\"gamma\":[-1094726528.0,-769562752.0,-84"
    "8000064.0],\"B\":[-3.9,0.0,-0.3],\"omega\":[352020896.0,3647927552.0,-1977"
    "987072.0],\"A\":[0.9,0.3,-1.3]},\"delay\":3.5e-09,\"a_lin\":0.9},\"0\":{\"scale"
    "\":1.0,\"parameter\":{\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\""
    "B\":[-3.9813032150268555,0.08622030913829803,-0.3152860999107361],\"omeg"
    "a\":[352020896.0,3647927552.0,-1977987072.0],\"A\":[0.9772450923919678,0."
    "3354335129261017,-1.312678575515747]},\"delay\":3.783458418887631e-09,\"a"
    "_lin\":0.27}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Gaussian Decomposition\",\"properties\":{\"range"
    "_offset_sec_high_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"ra"
    "nge_offset_sec_low_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\""
    "amplitude_lsb_high_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\""
    "amplitude_lsb_low_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"a"
    "mplitude_db\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"}},\"description\""
    ":\"riegl.gaussian_decomposition contains information relevant for "
    "extracting calibrated amplitudes and calibrated ranges from a Gaussian"
    " decomposition of full waveform data. This information is contained in"
    " a table with five columns. Two columns are to be used as input: "
    "amplitude_lsb_low_power and amplitude_lsb_high_power. The other three "
    "columns provide the outputs. Amplitude_db gives the calibrated "
    "amplitude in the optical regime in decibels. The range offset columns "
    "provide additive range offsets, given in units of seconds, for each ch"
    "annel.\",\"required\":[\"amplitude_lsb_low_power\",\"amplitude_lsb_high_powe"
    "r\",\"amplitude_db\",\"range_offset_sec_low_power\",\"range_offset_sec_high_"
    "power\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"range_offset_sec_high_power\":[],\"range_offset_sec_low_power\":[],\"amp"
    "litude_lsb_high_power\":[],\"amplitude_lsb_low_power\":[],\"amplitude_db\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Point cloud georeferencing "
    "information\",\"type\":\"object\",\"title\":\"Geo "
    "Tag\",\"properties\":{\"pose\":{\"description\":\"Coordinate Transformation "
    "Matrix to transform from File Coordinate System to Global Coordinate "
    "Reference System. 4x4 matrix stored as two dimensional array, row "
    "major order.\",\"items\":{\"description\":\"rows\",\"items\":{\"description\":\"co"
    "lumns\",\"type\":\"number\"},\"type\":\"array\",\"maxItems\":4,\"minItems\":4},\"typ"
    "e\":\"array\",\"maxItems\":4,\"minItems\":4},\"crs\":{\"description\":\"Global "
    "Coordinate Reference System. Please note that only 3D Cartesian "
    "Coordinate Systems are "
    "allowed.\",\"type\":\"object\",\"properties\":{\"epsg\":{\"description\":\"EPSG "
    "code\",\"type\":\"integer\",\"minimum\":0},\"wkt\":{\"description\":\"\\\"Well-Known"
    " Text\\\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta"
    "ndards/wkt-crs)\",\"type\":\"string\"},\"name\":{\"description\":\"Coordinate "
    "reference system name\",\"type\":\"string\"}}}},\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,4063"
    "882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847,113"
    "8787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196],[0"
    ".0,0.0,0.0,1.0]],\"crs\":{\"epsg\":4978,\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\""
    ",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORIT"
    "Y[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich"
    "\\\",0.0000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.0"
    "0000000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AX"
    "IS[\\\"Y\\\",EAST],AXIS[\\\"Z\\\",NORTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\",\"name\""
    ":\"WGS84 Geocentric\"}}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Geometric scale factor applied to point coordinates\",\""
    "exclusiveMinimum\":true,\"type\":\"number\",\"$schema\":\"http://json-schema.o"
    "rg/draft-04/schema#\",\"minimum\":0}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Parameters used for georeferencing of the point "
    "cloud\",\"type\":\"object\",\"title\":\"Georeferencing Parameters\",\"properties"
    "\":{\"socs_to_rocs_matrix\":{\"description\":\"Coordinate Transformation "
    "Matrix to transform from Scanner's Own Coordinate System to Record "
    "Coordinate System. 4x4 matrix stored as two dimensional array, row "
    "major order.\",\"items\":{\"description\":\"rows\",\"items\":{\"description\":\"co"
    "lumns\",\"type\":\"number\"},\"type\":\"array\",\"maxItems\":4,\"minItems\":4},\"typ"
    "e\":\"array\",\"maxItems\":4,\"minItems\":4},\"trajectory_offsets\":{\"descripti"
    "on\":\"Correction offsets applied to the trajectory data\",\"type\":\"object"
    "\",\"properties\":{\"offset_yaw\":{\"description\":\"[deg]\",\"type\":\"number\"},\""
    "offset_pitch\":{\"description\":\"[deg]\",\"type\":\"number\"},\"offset_height\":"
    "{\"description\":\"[m]\",\"type\":\"number\"},\"offset_north\":{\"description\":\"["
    "m]\",\"type\":\"number\"},\"offset_time\":{\"description\":\"[s]\",\"type\":\"number"
    "\"},\"offset_roll\":{\"description\":\"[deg]\",\"type\":\"number\"},\"offset_east\""
    ":{\"description\":\"[m]\",\"type\":\"number\"},\"version\":{\"description\":\"Meani"
    "ng of offset values and how to apply them; version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\",\"type\":\"integer\"}}},\"socs_to_body_matrix\":{\"descrip"
    "tion\":\"Coordinate Transformation Matrix to transform from Scanner's "
    "Own Coordinate System to Body Coordinate System. 4x4 matrix stored as "
    "two dimensional array, row major order.\",\"items\":{\"description\":\"rows\""
    ",\"items\":{\"description\":\"columns\",\"type\":\"number\"},\"type\":\"array\",\"max"
    "Items\":4,\"minItems\":4},\"type\":\"array\",\"maxItems\":4,\"minItems\":4},\"body"
    "_coordinate_system_type\":{\"description\":\"BODY coordinate frame (NED: "
    "North-East-Down, ENU: East-North-Up), default: NED\",\"type\":\"string\",\"e"
    "num\":[\"NED\",\"ENU\"]},\"trajectory_file\":{\"description\":\"Trajectory data "
    "used for georeferencing of the point cloud\",\"type\":\"object\",\"required\""
    ":[\"file_extension\"],\"properties\":{\"file_uuid\":{\"description\":\"File's "
    "Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Trajectory "
    "file extension, without the leading dot\",\"type\":\"string\"}}}},\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"body_coordinate_system_type\":\"NED\",\"trajectory_offsets\":{\"offset_yaw"
    "\":-0.45,\"offset_pitch\":0.01,\"offset_height\":-0.2,\"offset_north\":0.07,\""
    "offset_time\":18.007,\"offset_roll\":0.03,\"offset_east\":0.15,\"version\":0}"
    ",\"socs_to_body_matrix\":[[-0.269827776749716,-0.723017716139738,0.63595"
    "4678449952,0.0],[0.962908599449764,-0.20260517250352,0.178208229833847"
    ",0.0],[0.0,0.660451759194338,0.7508684796801,0.0],[0.0,0.0,0.0,1.0]],\""
    "trajectory_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\",\""
    "file_extension\":\"pofx\"}}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about GNSS "
    "processing\",\"type\":\"object\",\"title\":\"GNSS Processing "
    "Information\",\"properties\":{\"base_stations\":{\"description\":\"List of "
    "GNSS base stations\",\"items\":{\"description\":\"GNSS base station details\""
    ",\"type\":\"object\",\"required\":[\"latitude\",\"longitude\",\"altitude\"],\"prope"
    "rties\":{\"altitude\":{\"description\":\"Altitude "
    "[m]\",\"type\":\"number\"},\"latitude\":{\"description\":\"Latitude "
    "[deg]\",\"type\":\"number\"},\"longitude\":{\"description\":\"Longitude [deg]\",\""
    "type\":\"number\"}}},\"type\":\"array\"}},\"$schema\":\"http://json-schema.org/d"
    "raft-04/schema#\"}"
    ,
    /* example: */
    "{\"base_stations\":[{\"altitude\":379.8004150390625,\"latitude\":48.65839116"
    "2,\"longitude\":15.663389822}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Item Names\",\"description\":\"Map of item "
    "names\",\"patternProperties\":{\"^-?[0-9]+$\":{\"description\":\"One field per"
    " item, field name is item id, field value is item name\",\"type\":\"string"
    "\"}},\"additionalProperties\":false,\"$schema\":\"http://json-schema.org/dra"
    "ft-04/schema#\"}"
    ,
    /* example: */
    "{\"1\":\"Name of item with id 1\",\"2\":\"Name of item with id 2\",\"47\":\"Name "
    "of item with id 47\",\"-1\":\"Name of item with id -1\"}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Software License "
    "Keys\",\"description\":\"License keys for software "
    "features\",\"patternProperties\":{\"^.*$\":{\"description\":\"Each field of "
    "the object represents a feature and holds a list of license keys, "
    "where the field name is the feature "
    "name.\",\"items\":{\"description\":\"License key (example: '46AE032A - "
    "39882AC4 - 9EC0A184 - 6F163D73')\",\"type\":\"string\"},\"type\":\"array\",\"min"
    "Items\":1}},\"additionalProperties\":false,\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"Full Waveform Analysis Topography with GPU support\":[\"8AB44126 - "
    "23B92250 - 16E2689F - 34EF7E7B\"],\"MTA resolution\":[\"468E020A - "
    "39A922E4 - B681A184 - 673E3D72\"],\"Full Waveform Analysis "
    "Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA - "
    "8D2ED557\"],\"Georeferencing\":[\"46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Parameters for MTA "
    "processing\",\"type\":\"object\",\"title\":\"MTA "
    "Settings\",\"properties\":{\"modulation_depth\":{\"description\":\"Depth of "
    "pulse position modulation in meter.\",\"type\":\"number\",\"minimum\":0},\"zon"
    "e_count\":{\"description\":\"Maximum number of MTA zones.\",\"type\":\"integer"
    "\",\"maximum\":255,\"minimum\":0},\"zone_width\":{\"description\":\"Width of a "
    "MTA zone in meter.\",\"type\":\"number\",\"minimum\":0}},\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"modulation_depth\":9.368514,\"zone_count\":23,\"zone_width\":149.896225}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Near Range Correction "
    "Table\",\"properties\":{\"content\":{\"description\":\"Correction value [m] to"
    " be added to the raw range\",\"items\":{\"type\":\"number\"},\"type\":\"array\",\""
    "maxItems\":2000,\"minItems\":1},\"delta\":{\"description\":\"Delta between "
    "table entries [m], first entry is at range = 0 "
    "m\",\"type\":\"number\"}},\"description\":\"Lookup table for range correction "
    "based on raw range\",\"required\":[\"delta\",\"content\"],\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"content\":[0.0],\"delta\":0.512}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Noise "
    "Estimates\",\"properties\":{\"amplitude\":{\"description\":\"Amplitude [dB]\",\""
    "items\":{\"type\":\"number\"},\"type\":\"array\"},\"range_sigma\":{\"description\":"
    "\"Sigma range [m]\",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"amplitude"
    "_sigma\":{\"description\":\"Sigma amplitude [dB]\",\"items\":{\"type\":\"number\""
    "},\"type\":\"array\"}},\"description\":\"Standard deviation for range and "
    "amplitude as a function of amplitude\",\"required\":[\"amplitude\",\"range_s"
    "igma\",\"amplitude_sigma\"],\"$schema\":\"http://json-schema.org/draft-04/sc"
    "hema#\"}"
    ,
    /* example: */
    "{\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],\"range_sig"
    "ma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,0.016,0.014"
    "],\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0."
    "432,0.39,0.354]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Notch "
    "Filter\",\"properties\":{\"range_minimum\":{\"description\":\"Minimum range "
    "[m]\",\"type\":\"number\"},\"amplitude_maximum\":{\"description\":\"Maximum "
    "amplitude [dB]\",\"type\":\"number\",\"minimum\":0},\"range_maximum\":{\"descrip"
    "tion\":\"Maximum range [m]\",\"type\":\"number\"}},\"description\":\"Notch "
    "filter parameters for window glass echoes\",\"required\":[\"range_minimum\""
    ",\"range_maximum\",\"amplitude_maximum\"],\"$schema\":\"http://json-schema.or"
    "g/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"range_minimum\":-0.5,\"amplitude_maximum\":18.0,\"range_maximum\":0.2}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pixel "
    "Information\",\"properties\":{\"size_llcs\":{\"description\":\"Size of pixels "
    "in a locally levelled cartesian coordinate system (xy). This is only "
    "used for pixels based on a map projection.\",\"$ref\":\"#/definitions/pixe"
    "l_size\"},\"size\":{\"description\":\"Size of pixels in file coordinate "
    "system.\",\"$ref\":\"#/definitions/pixel_size\"}},\"description\":\"Details "
    "about the pixels contained in the "
    "file\",\"definitions\":{\"pixel_size\":{\"description\":\"Size of "
    "pixels.\",\"items\":{\"description\":\"Length of pixel edge [m].\",\"type\":\"nu"
    "mber\",\"minimum\":0},\"type\":\"array\",\"maxItems\":2,\"minItems\":2}},\"require"
    "d\":[\"size\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"size_llcs\":[0.5156575252891171,0.5130835356683303],\"size\":[0.5971642"
    "834779395,0.5971642834779395]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Plane Patch Matching\",\"properties\":{\"plane_p"
    "atch_file_one\":{\"description\":\"Reference to the plane patch file one\","
    "\"$ref\":\"#/definitions/file_reference\"},\"plane_patch_file_two\":{\"descri"
    "ption\":\"Reference to the plane patch file "
    "two\",\"$ref\":\"#/definitions/file_reference\"}},\"description\":\"Details "
    "about the plane patch matching "
    "process\",\"definitions\":{\"file_reference\":{\"description\":\"Reference to "
    "a plane patch file\",\"type\":\"object\",\"required\":[\"file_uuid\",\"file_path"
    "\"],\"properties\":{\"file_path\":{\"description\":\"Path of the plane patch "
    "file relative to the match "
    "file\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"Plane patch file's "
    "Universally Unique Identifier (RFC 4122)\",\"type\":\"string\"}}}},\"require"
    "d\":[\"plane_patch_file_one\",\"plane_patch_file_two\"],\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"plane_patch_file_one\":{\"file_path\":\"Record009_Line001/191025_121410_"
    "Scanner_1.ptch\",\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\"},\"p"
    "lane_patch_file_two\":{\"file_path\":\"project.ptch\",\"file_uuid\":\"fa47d509"
    "-a64e-49ce-8b14-ff3130fbefa9\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Statistics about plane patches found by plane patch "
    "extractor\",\"type\":\"object\",\"title\":\"Plane Patch "
    "Statistics\",\"properties\":{\"total_area\":{\"description\":\"sum of all "
    "plane patch areas [m\\u00b2]\",\"type\":\"number\"},\"total_horizontal_area\":"
    "{\"description\":\"sum of all plane patch areas projected to horizontal "
    "plane [m\\u00b2]\",\"type\":\"number\"}},\"$schema\":\"http://json-schema.org/d"
    "raft-04/schema#\"}"
    ,
    /* example: */
    "{\"total_area\":14007.965,\"total_horizontal_area\":13954.601}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Plane Slope Class "
    "Info\",\"properties\":{\"classes\":{\"description\":\"Class definition "
    "table\",\"patternProperties\":{\"^[0-9]+$\":{\"description\":\"one field per "
    "class, field name is class number, field value is class name\",\"type\":\""
    "string\"}},\"type\":\"object\",\"additionalProperties\":false},\"settings\":{\"d"
    "escription\":\"Classification settings, details see documentation of rdb"
    "planes\",\"type\":\"object\",\"oneOf\":[{\"$ref\":\"#/definitions/method-1\"},{\"$"
    "ref\":\"#/definitions/method-2\"}]}},\"description\":\"Settings and classes "
    "for plane slope classification\",\"definitions\":{\"method-2\":{\"descriptio"
    "n\":\"Classification method 2\",\"type\":\"object\",\"required\":[\"plane_classi"
    "fication_method\",\"sloping_plane_classes_minimum_angle\",\"sloping_plane_"
    "classes_maximum_angle\"],\"properties\":{\"sloping_plane_classes_minimum_a"
    "ngle\":{\"description\":\"minimum inclination angle of sloping plane "
    "patches [deg]\",\"type\":\"number\",\"maximum\":360.0,\"minimum\":-360.0},\"plan"
    "e_classification_method\":{\"description\":\"method ID (=2)\",\"type\":\"integ"
    "er\",\"maximum\":2,\"minimum\":2},\"sloping_plane_classes_maximum_angle\":{\"d"
    "escription\":\"maximum inclination angle of sloping plane patches [deg]\""
    ",\"type\":\"number\",\"maximum\":360.0,\"minimum\":-360.0}}},\"method-1\":{\"desc"
    "ription\":\"Classification method 1\",\"type\":\"object\",\"required\":[\"plane_"
    "classification_method\",\"maximum_inclination_angle_horizontal\"],\"proper"
    "ties\":{\"maximum_inclination_angle_horizontal\":{\"description\":\"maximum "
    "inclination angle of horizontal plane patches [deg]\",\"type\":\"number\",\""
    "maximum\":360.0,\"minimum\":-360.0},\"plane_classification_method\":{\"descr"
    "iption\":\"method ID (=1)\",\"type\":\"integer\",\"maximum\":1,\"minimum\":1}}}},"
    "\"required\":[\"settings\",\"classes\"],\"$schema\":\"http://json-schema.org/dr"
    "aft-04/schema#\"}"
    ,
    /* example: */
    "{\"classes\":{\"9\":\"vertical, pointing west\",\"8\":\"vertical, pointing "
    "north\",\"14\":\"horizontal, pointing down\",\"7\":\"vertical, pointing "
    "south\",\"12\":\"sloping, pointing down and north\",\"6\":\"vertical, pointing"
    " east\",\"13\":\"sloping, pointing down and west\",\"5\":\"sloping, pointing "
    "up and west\",\"4\":\"sloping, pointing up and north\",\"1\":\"horizontal, "
    "pointing up\",\"3\":\"sloping, pointing up and south\",\"2\":\"sloping, "
    "pointing up and east\",\"10\":\"sloping, pointing down and "
    "east\",\"11\":\"sloping, pointing down and south\"},\"settings\":{\"sloping_pl"
    "ane_classes_minimum_angle\":10.0,\"plane_classification_method\":2,\"slopi"
    "ng_plane_classes_maximum_angle\":70.0}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Point Attribute "
    "Groups\",\"description\":\"Grouping and sorting of point attributes for "
    "visualization "
    "purposes\",\"patternProperties\":{\"^.*$\":{\"description\":\"Each field of "
    "the object represents a point attribute group and holds a list of "
    "point attributes, where the field name is the group "
    "name.\",\"items\":{\"description\":\"Point attribute full name or name "
    "pattern (perl regular expression syntax)\",\"type\":\"string\"},\"type\":\"arr"
    "ay\",\"minItems\":1}},\"additionalProperties\":false,\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"Coordinates/Vectors\":[\"riegl.xyz\",\"riegl.range\",\"riegl.theta\",\"riegl"
    ".phi\"],\"Other Attributes\":[\"riegl.selected\",\"riegl.visible\"],\"Time\":[\""
    "riegl.timestamp\"],\"Secondary "
    "Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available\"],\"Primary"
    " "
    "Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.deviation\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about point cloud "
    "files\",\"type\":\"object\",\"title\":\"Point Cloud Information\",\"properties\":"
    "{\"comments\":{\"description\":\"Comments\",\"type\":\"string\"},\"field_of_appli"
    "cation\":{\"description\":\"Field of application\",\"type\":\"string\",\"enum\":["
    "\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"]},\"project\":"
    "{\"description\":\"Project name\",\"type\":\"string\"}},\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"comments\":\"Line 3\",\"field_of_application\":\"ALS\",\"project\":\"Campaign "
    "4\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pose "
    "Estimation\",\"properties\":{\"position\":{\"description\":\"Position "
    "coordinates and position accuracy values as measured by GNSS in the "
    "specified Coordinate Reference System (CRS)\",\"type\":\"object\",\"required"
    "\":[\"coordinate_1\",\"coordinate_2\",\"coordinate_3\",\"horizontal_accuracy\","
    "\"vertical_accuracy\",\"crs\"],\"properties\":{\"vertical_accuracy\":{\"descrip"
    "tion\":\"Vertical accuracy [m]\",\"exclusiveMinimum\":true,\"type\":\"number\","
    "\"minimum\":0},\"crs\":{\"description\":\"Global Coordinate Reference System\""
    ",\"type\":\"object\",\"required\":[\"epsg\"],\"properties\":{\"epsg\":{\"descriptio"
    "n\":\"EPSG "
    "code\",\"type\":\"integer\",\"minimum\":0},\"wkt\":{\"description\":\"\\\"Well-Known"
    " Text\\\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta"
    "ndards/wkt-crs)\",\"type\":\"string\"}}},\"horizontal_accuracy\":{\"descriptio"
    "n\":\"Horizontal accuracy [m]\",\"exclusiveMinimum\":true,\"type\":\"number\",\""
    "minimum\":0},\"coordinate_1\":{\"description\":\"Coordinate 1 as defined by "
    "axis 1 of the specified CRS (e.g., X, "
    "Latitude)\",\"type\":\"number\"},\"coordinate_2\":{\"description\":\"Coordinate "
    "2 as defined by axis 2 of the specified CRS (e.g., Y, "
    "Longitude)\",\"type\":\"number\"},\"coordinate_3\":{\"description\":\"Coordinate"
    " 3 as defined by axis 3 of the specified CRS (e.g., Z, Altitude)\",\"typ"
    "e\":\"number\"}}},\"barometric_height_amsl\":{\"description\":\"Altitude "
    "determined based on the atmospheric pressure according to the standard"
    " atmosphere laws "
    "[m].\",\"type\":\"number\"},\"orientation\":{\"description\":\"Orientation "
    "values and orientation accuracies, measured with IMU or inclination se"
    "nsors.\",\"type\":\"object\",\"required\":[\"roll\",\"pitch\",\"yaw\",\"roll_accurac"
    "y\",\"pitch_accuracy\",\"yaw_accuracy\"],\"properties\":{\"roll\":{\"description"
    "\":\"Roll angle about scanner X-axis [deg]\",\"type\":\"number\",\"maximum\":36"
    "0,\"minimum\":-360},\"yaw_accuracy\":{\"description\":\"Yaw angle accuracy [d"
    "eg]\",\"exclusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},\"yaw\":{\"desc"
    "ription\":\"Yaw angle about scanner Z-axis [deg]\",\"type\":\"number\",\"maxim"
    "um\":360,\"minimum\":-360},\"pitch_accuracy\":{\"description\":\"Pitch angle "
    "accuracy [deg]\",\"exclusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},\""
    "roll_accuracy\":{\"description\":\"Roll angle accuracy [deg]\",\"exclusiveMi"
    "nimum\":true,\"type\":\"number\",\"minimum\":0},\"pitch\":{\"description\":\"Pitch"
    " angle about scanner Y-axis [deg]\",\"type\":\"number\",\"maximum\":360,\"mini"
    "mum\":-360}}}},\"description\":\"Estimated position and orientation "
    "information as measured by GNSS, IMU or inclination sensors\",\"required"
    "\":[\"orientation\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"position\":{\"vertical_accuracy\":1.3314999341964722,\"crs\":{\"epsg\":4979"
    ",\"wkt\":\"GEOGCS[\\\"WGS84 / Geographic\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84"
    "\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\"
    "\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000000000000,AUTHORITY[\\"
    "\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Degree\\\",0.01745329251994329547,AUTHORITY[\\\""
    "EPSG\\\",\\\"9102\\\"]],AXIS[\\\"Latitude\\\",NORTH],AXIS[\\\"Longitude\\\",EAST],AU"
    "THORITY[\\\"EPSG\\\",\\\"4979\\\"]]\"},\"horizontal_accuracy\":0.810699999332428,"
    "\"coordinate_1\":48.655799473,\"coordinate_2\":15.645033406,\"coordinate_3\""
    ":362.7124938964844},\"barometric_height_amsl\":386.7457796227932,\"orient"
    "ation\":{\"roll\":3.14743073066123,\"yaw_accuracy\":1.0094337839368757,\"yaw"
    "\":101.87293630292045,\"pitch_accuracy\":0.009433783936875745,\"roll_accur"
    "acy\":0.009433783936875745,\"pitch\":1.509153024827064}}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pose "
    "Sensors\",\"properties\":{\"gyroscope\":{\"description\":\"Gyroscope details\","
    "\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\""
    ",\"origin\",\"relative_nonlinearity\"],\"properties\":{\"offset\":{\"descriptio"
    "n\":\"Value to be subtracted from raw measurement values\",\"$ref\":\"#/defi"
    "nitions/vector\"},\"x_axis\":{\"description\":\"Sensitive X axis of sensor "
    "at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"origin\":{\"description\":\"Sensor "
    "origin in SOCS [m] at frame angle = 0\",\"$ref\":\"#/definitions/vector\"},"
    "\"relative_nonlinearity\":{\"description\":\"Relative nonlinearity, dimensi"
    "onless\",\"$ref\":\"#/definitions/vector\"},\"unit\":{\"description\":\"Unit of "
    "raw data and calibration values, 1 LSB in rad/s\",\"exclusiveMinimum\":tr"
    "ue,\"type\":\"number\",\"minimum\":0},\"z_axis\":{\"description\":\"Sensitive Z "
    "axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"y_axis\":{\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\",\"$ref\":\"#/definitions/vector\"}}},\""
    "magnetic_field_sensor\":{\"description\":\"Magnetic Field Sensor details\","
    "\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\""
    ",\"fixed\",\"relative_nonlinearity\"],\"properties\":{\"offset\":{\"description"
    "\":\"Value to be subtracted from raw measurement values\",\"$ref\":\"#/defin"
    "itions/vector\"},\"x_axis\":{\"description\":\"Sensitive X axis of sensor at"
    " frame angle = 0\",\"$ref\":\"#/definitions/vector\"},\"relative_nonlinearit"
    "y\":{\"description\":\"Relative nonlinearity, dimensionless\",\"$ref\":\"#/def"
    "initions/vector\"},\"unit\":{\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in nT\",\"exclusiveMinimum\":true,\"type\":\"numbe"
    "r\",\"minimum\":0},\"z_axis\":{\"description\":\"Sensitive Z axis of sensor at"
    " frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"y_axis\":{\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"fixed\":{\"description\":\"Distortion "
    "of magnetic field caused by non-rotating scanner part\",\"$ref\":\"#/defin"
    "itions/vector\"}}},\"accelerometer\":{\"description\":\"Accelerometer detail"
    "s\",\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offs"
    "et\",\"origin\",\"relative_nonlinearity\"],\"properties\":{\"offset\":{\"descrip"
    "tion\":\"Value to be subtracted from raw measurement values\",\"$ref\":\"#/d"
    "efinitions/vector\"},\"x_axis\":{\"description\":\"Sensitive X axis of "
    "sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"origin\":{\"description\":\"Sensor "
    "origin in SOCS [m] at frame angle = 0\",\"$ref\":\"#/definitions/vector\"},"
    "\"relative_nonlinearity\":{\"description\":\"Relative nonlinearity, dimensi"
    "onless\",\"$ref\":\"#/definitions/vector\"},\"unit\":{\"description\":\"Unit of "
    "raw data and calibration values, 1 LSB in 9.81 m/s\\u00b2\",\"exclusiveMi"
    "nimum\":true,\"type\":\"number\",\"minimum\":0},\"z_axis\":{\"description\":\"Sens"
    "itive Z axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"y_axis\":{\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"}}}},\"description\":\"Details on "
    "position and orientation "
    "sensors\",\"definitions\":{\"vector\":{\"items\":{\"description\":\"Index 0=X, "
    "1=Y, 2=Z component\",\"type\":\"number\"},\"type\":\"array\",\"maxItems\":3,\"minI"
    "tems\":3}},\"required\":[\"gyroscope\",\"accelerometer\",\"magnetic_field_sens"
    "or\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"gyroscope\":{\"offset\":[-50.92609786987305,146.15643310546875,62.43272"
    "78137207],\"x_axis\":[-121.195556640625,0.8219714164733887,0.23130317032"
    "33719],\"origin\":[0.026900000870227814,-0.03999999910593033,-0.08950000"
    "256299973],\"relative_nonlinearity\":[2.888176311444113e-07,1.0627416457"
    "9645e-07,-1.7186295080634935e-39],\"unit\":0.00014544410805683583,\"z_axi"
    "s\":[0.555869996547699,119.22135162353516,0.467585027217865],\"y_axis\":["
    "-0.440765917301178,-0.7897399663925171,119.5894775390625]},\"magnetic_f"
    "ield_sensor\":{\"offset\":[-23812.052734375,5606.57666015625,2493.28125],"
    "\"x_axis\":[-0.011162743903696537,-2.315962774446234e-05,0.0001681884459"
    "6855342],\"relative_nonlinearity\":[0.0,0.0,0.0],\"unit\":91.7431182861328"
    "1,\"z_axis\":[0.00041987866279669106,7.876977906562388e-05,0.01140710432"
    "0824146],\"y_axis\":[0.00027888521435670555,-0.011427424848079681,-5.204"
    "829722060822e-05],\"fixed\":[-1576.010498046875,1596.081787109375,0.0]},"
    "\"accelerometer\":{\"offset\":[-733.3636474609375,58.969032287597656,1060."
    "2550048828125],\"x_axis\":[-15008.123046875,56.956390380859375,-60.51756"
    "66809082],\"origin\":[0.026900000870227814,-0.03999999910593033,-0.08950"
    "000256299973],\"relative_nonlinearity\":[0.0,0.0,0.0],\"unit\":6.666666740"
    "784422e-05,\"z_axis\":[1.639882206916809,15166.744140625,-116.9974288940"
    "4297],\"y_axis\":[-7.027288913726807,-44.12333679199219,14952.3701171875"
    "]}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pulse Position "
    "Modulation\",\"properties\":{\"code_phase_mode\":{\"description\":\"0: no "
    "synchronization, 1: toggle between 2 phases, 2: increment with phase_i"
    "ncrement\",\"type\":\"integer\",\"maximum\":255,\"minimum\":0},\"pulse_interval\""
    ":{\"description\":\"Explicit table of the pulse position modulation used "
    "for MTA resolution. Table gives times between successive laser pulses "
    "in seconds.\",\"items\":{\"type\":\"number\",\"minimum\":0},\"type\":\"array\"},\"nu"
    "m_mod_ampl\":{\"description\":\"Number of different modulation amplitudes "
    "(2: binary modulation)\",\"type\":\"integer\",\"maximum\":255,\"minimum\":0},\"p"
    "hase_step\":{\"description\":\"Step width in phase of modulation code from"
    " line to line\",\"type\":\"integer\",\"maximum\":255,\"minimum\":0},\"length\":{\""
    "description\":\"Length of code\",\"type\":\"integer\",\"maximum\":255,\"minimum\""
    ":0}},\"description\":\"Laser pulse position modulation used for MTA resol"
    "ution\",\"required\":[\"length\",\"num_mod_ampl\",\"pulse_interval\"],\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"code_phase_mode\":2,\"pulse_interval\":[2.759375e-06,2.759375e-06,2.759"
    "375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-"
    "06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2."
    "821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.75937"
    "5e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06"
    ",2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.75"
    "9375e-06,2.821875e-06],\"num_mod_ampl\":2,\"phase_step\":5,\"length\":31}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Range "
    "Statistics\",\"properties\":{\"average\":{\"description\":\"Average "
    "value\",\"type\":\"number\"},\"maximum\":{\"description\":\"Maximum "
    "value\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum "
    "value\",\"type\":\"number\"},\"std_dev\":{\"description\":\"Standard "
    "deviation\",\"type\":\"number\"}},\"description\":\"Statistics about target "
    "distance wrt. SOCS origin\",\"required\":[\"minimum\",\"average\",\"maximum\",\""
    "std_dev\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"average\":15.49738,\"maximum\":574.35,\"minimum\":0.919,\"std_dev\":24.349}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Receiver "
    "Internals\",\"properties\":{\"ns\":{\"description\":\"Number of samples\",\"type"
    "\":\"integer\",\"maximum\":4095,\"minimum\":0},\"si\":{\"description\":\"Start "
    "index (hw_start)\",\"type\":\"number\",\"maximum\":255,\"minimum\":0},\"a\":{\"ite"
    "ms\":{\"type\":\"number\"},\"description\":\"Amplitude [dB]\",\"type\":\"array\",\"m"
    "axItems\":256,\"minItems\":1},\"sr\":{\"description\":\"Sample rate [Hz]\",\"exc"
    "lusiveMinimum\":true,\"type\":\"number\",\"minimum\":0},\"nt\":{\"description\":\""
    "Number of traces\",\"type\":\"integer\",\"maximum\":255,\"minimum\":0},\"tbl\":{\""
    "items\":{\"$ref\":\"#/definitions/fp_table\"},\"description\":\"various "
    "internal data\",\"type\":\"array\",\"minItems\":1},\"t\":{\"patternProperties\":{"
    "\"^[0-9]+$\":{\"description\":\"one field per channel, field name is "
    "channel index\",\"$ref\":\"#/definitions/fp\"}},\"type\":\"object\",\"additional"
    "Properties\":false},\"mw\":{\"description\":\"Maximum weight\",\"exclusiveMini"
    "mum\":true,\"type\":\"number\",\"maximum\":4095,\"minimum\":0},\"ex\":{\"descripti"
    "on\":\"DEPRECATED, use 'riegl.exponential_decomposition' "
    "instead\",\"type\":\"object\"}},\"description\":\"Receiver "
    "Internals\",\"definitions\":{\"fp\":{\"description\":\"Fingerprint values\",\"ty"
    "pe\":\"object\",\"required\":[\"s\",\"w\"],\"properties\":{\"w\":{\"items\":{\"items\":"
    "{\"type\":\"number\"},\"type\":\"array\",\"maxItems\":5,\"minItems\":5},\"type\":\"ar"
    "ray\",\"maxItems\":256,\"minItems\":1},\"s\":{\"items\":{\"items\":{\"type\":\"numbe"
    "r\"},\"type\":\"array\",\"maxItems\":4096,\"minItems\":1},\"type\":\"array\",\"maxIt"
    "ems\":256,\"minItems\":1}}},\"fp_table_row\":{\"items\":{\"type\":\"number\"},\"ty"
    "pe\":\"array\",\"maxItems\":2048,\"minItems\":1},\"fp_table\":{\"required\":[\"ch\""
    ",\"tc\",\"nx\",\"ny\",\"tv\"],\"type\":\"object\",\"desription\":\"scanner internal d"
    "ata\",\"properties\":{\"tv\":{\"items\":{\"oneOf\":[{\"$ref\":\"#/definitions/fp_t"
    "able_row\"},{\"type\":\"number\"}]},\"type\":\"array\",\"maxItems\":2048,\"minItem"
    "s\":1},\"ny\":{\"description\":\"number of y entries\",\"type\":\"integer\",\"min\""
    ":1,\"max\":2048},\"tc\":{\"description\":\"table data type "
    "code\",\"type\":\"integer\",\"min\":0,\"max\":255},\"nx\":{\"description\":\"number "
    "of x entries\",\"type\":\"integer\",\"min\":1,\"max\":2048},\"ch\":{\"description\""
    ":\"channel number\",\"type\":\"integer\",\"min\":0,\"max\":255}}}},\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"ns\":400,\"si\":48,\"a\":[-1.55],\"sr\":7959997000.0,\"nt\":128,\"tbl\":[{\"tv\":"
    "[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.5]],\"ny\":2,\"tc\":1,\"nx\":5,\"ch\":0}],\"t\":"
    "{\"1\":{\"w\":[[78,86,126,134,31],[78,86,126,134,31]],\"s\":[[1.23,4.56],[7."
    "89,0.12]]},\"0\":{\"w\":[[78,86,126,134,31],[78,86,126,134,31]],\"s\":[[1.23"
    ",4.56],[7.89,0.12]]}},\"mw\":31}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Reflectance Calculation "
    "Table\",\"properties\":{\"content\":{\"description\":\"Correction value [dB] "
    "to be added to the amplitude\",\"items\":{\"type\":\"number\"},\"type\":\"array\""
    ",\"maxItems\":2000,\"minItems\":1},\"delta\":{\"description\":\"Delta between "
    "table entries [m], first entry is at range = 0 "
    "m\",\"type\":\"number\"}},\"description\":\"Lookup table for reflectance "
    "calculation based on amplitude and range\",\"required\":[\"delta\",\"content"
    "\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"content\":[-33.01],\"delta\":0.150918}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Near-range reflectance correction\",\"properti"
    "es\":{\"reflectance_correction_db\":{\"description\":\"Near range "
    "reflectance correction in dB as a function of range over "
    "angle\",\"items\":{\"description\":\"rows (each array corresponds to a "
    "range)\",\"items\":{\"description\":\"columns (each value corresponds to an "
    "angle)\",\"type\":\"number\"},\"type\":\"array\"},\"type\":\"array\"},\"ranges_m\":{\""
    "description\":\"Range [m]\",\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"li"
    "ne_angles_deg\":{\"description\":\"Angle [deg]\",\"items\":{\"type\":\"number\"},"
    "\"type\":\"array\"}},\"description\":\"Range-dependent and "
    "scan-angle-dependent correction of reflectance reading\",\"required\":[\"r"
    "anges_m\",\"line_angles_deg\",\"reflectance_correction_db\"],\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0."
    "01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0"
    ".3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01]],\"ra"
    "nges_m\":[0.0,1.0,2.0,3.0],\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2.0,2"
    ".5,3.0,3.5,4.0]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"RiLOC Processing "
    "Information\",\"properties\":{\"optimizer\":{\"description\":\"Optimization "
    "result information and statistics.\",\"type\":\"object\",\"properties\":{}},\""
    "trajectory\":{\"description\":\"Trajectory information and statistics.\",\"t"
    "ype\":\"object\",\"properties\":{}},\"georeferencing\":{\"description\":\"Intern"
    "al georeferencing parameters.\",\"type\":\"object\",\"properties\":{}},\"plane"
    "s\":{\"description\":\"Information on plane correspondences and used "
    "extraction/matching "
    "templates.\",\"type\":\"object\",\"properties\":{}}},\"description\":\"Details "
    "about RiLOC processing\",\"required\":[\"trajectory\",\"optimizer\",\"georefer"
    "encing\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"optimizer\":{},\"trajectory\":{},\"georeferencing\":{},\"planes\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scan pattern "
    "description\",\"definitions\":{\"program\":{\"description\":\"Measurement prog"
    "ram\",\"type\":\"object\",\"required\":[\"name\"],\"properties\":{\"laser_prr\":{\"d"
    "escription\":\"Laser Pulse Repetition Rate [Hz]\",\"exclusiveMinimum\":fals"
    "e,\"type\":\"number\",\"minimum\":0},\"name\":{\"description\":\"Name of "
    "measurement program\",\"type\":\"string\"}}}},\"title\":\"Scan "
    "Pattern\",\"properties\":{\"segments\":{\"description\":\"Segmented Line Scan "
    "Pattern\",\"type\":\"object\",\"required\":[\"list\"],\"properties\":{\"program\":{"
    "\"$ref\":\"#/definitions/program\"},\"list\":{\"items\":{\"description\":\"Line "
    "Scan Segment\",\"type\":\"object\",\"required\":[\"start\",\"stop\",\"increment\"],"
    "\"properties\":{\"start\":{\"description\":\"Start angle in SOCS [deg]\",\"type"
    "\":\"number\",\"maximum\":360.0,\"minimum\":0.0},\"increment\":{\"description\":\""
    "Increment of angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0,\"mini"
    "mum\":0.0},\"stop\":{\"description\":\"Stop angle in SOCS [deg]\",\"type\":\"num"
    "ber\",\"maximum\":720.0,\"minimum\":0.0}}},\"type\":\"array\"}}},\"line\":{\"descr"
    "iption\":\"Line Scan Pattern\",\"type\":\"object\",\"required\":[\"start\",\"stop\""
    ",\"increment\"],\"properties\":{\"start\":{\"description\":\"Start angle in "
    "SOCS [deg]\",\"type\":\"number\",\"maximum\":360.0,\"minimum\":0.0},\"increment\""
    ":{\"description\":\"Increment of angle in SOCS [deg]\",\"type\":\"number\",\"ma"
    "ximum\":90.0,\"minimum\":0.0},\"program\":{\"$ref\":\"#/definitions/program\"},"
    "\"stop\":{\"description\":\"Stop angle in SOCS [deg]\",\"type\":\"number\",\"maxi"
    "mum\":720.0,\"minimum\":0.0}}},\"rectangular\":{\"description\":\"Rectangular "
    "Field Of View Scan Pattern\",\"type\":\"object\",\"required\":[\"phi_start\",\"p"
    "hi_stop\",\"phi_increment\",\"theta_start\",\"theta_stop\",\"theta_increment\"]"
    ",\"properties\":{\"phi_increment\":{\"description\":\"Increment of phi angle "
    "in SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0,\"minimum\":0.0},\"theta_st"
    "op\":{\"description\":\"Stop theta angle in SOCS [deg]\",\"type\":\"number\",\"m"
    "aximum\":180.0,\"minimum\":0.0},\"program\":{\"$ref\":\"#/definitions/program\""
    "},\"phi_start\":{\"description\":\"Start phi angle in SOCS [deg]\",\"type\":\"n"
    "umber\",\"maximum\":360.0,\"minimum\":0.0},\"theta_start\":{\"description\":\"St"
    "art theta angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":180.0,\"minimu"
    "m\":0.0},\"theta_increment\":{\"description\":\"Increment of theta angle in "
    "SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0,\"minimum\":0.0},\"phi_stop\":{"
    "\"description\":\"Stop phi angle in SOCS [deg]\",\"type\":\"number\",\"maximum\""
    ":720.0,\"minimum\":0.0}}}},\"$schema\":\"http://json-schema.org/draft-04/sc"
    "hema#\"}"
    ,
    /* example: */
    "{\"rectangular\":{\"phi_increment\":0.04,\"theta_stop\":130.0,\"program\":{\"la"
    "ser_prr\":100000.0,\"name\":\"High Speed\"},\"phi_start\":45.0,\"theta_start\":"
    "30.0,\"theta_increment\":0.04,\"phi_stop\":270.0}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about laser shot "
    "files\",\"type\":\"object\",\"title\":\"Shot Information\",\"properties\":{\"shot_"
    "file\":{\"type\":\"object\",\"required\":[\"file_extension\"],\"properties\":{\"fi"
    "le_uuid\":{\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Shot file "
    "extension, without the leading dot\",\"type\":\"string\"}}}},\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_uuid\":\"26a00815-67c0-4bff-8fe8-c577378fe663\",\"file"
    "_extension\":\"sodx\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Stored "
    "filters\",\"properties\":{\"activated\":{\"description\":\"Apply ('true') or "
    "ignore ('false') all "
    "filters\",\"type\":\"boolean\"},\"filters\":{\"description\":\"List of point "
    "filters\",\"items\":{\"description\":\"Point filter definition\",\"type\":\"obje"
    "ct\",\"required\":[\"activated\",\"title\",\"description\",\"filter\"],\"propertie"
    "s\":{\"description\":{\"description\":\"A brief description of the filter "
    "(e.g. for display in a "
    "tooltip)\",\"type\":\"string\"},\"filter\":{\"description\":\"The RDB filter "
    "string to apply (e.g. when reading points or index), details see "
    "documentation of function "
    "select()\",\"type\":\"string\"},\"activated\":{\"description\":\"Apply ('true') "
    "or ignore ('false') this "
    "filter\",\"type\":\"boolean\"},\"title\":{\"description\":\"A short filter title"
    " (e.g. for display in a selection "
    "list)\",\"type\":\"string\"}}},\"type\":\"array\"}},\"description\":\"Point "
    "filters applied in addition to the application-defined filters\",\"requi"
    "red\":[\"activated\",\"filters\"],\"$schema\":\"http://json-schema.org/draft-0"
    "4/schema#\"}"
    ,
    /* example: */
    "{\"activated\":true,\"filters\":[{\"description\":\"Ignore points with "
    "uncertain MTA zone assignment\",\"filter\":\"riegl.mta_uncertain_point == "
    "0\",\"activated\":false,\"title\":\"Ignore uncertain points from MTA "
    "resolution\"},{\"description\":\"Ignore points with an echo signal "
    "amplitude below the dynamic "
    "trigger\",\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\",\"activated\":false,\"title\":\"Ignore points below dynamic trigger "
    "level\"},{\"description\":\"Ignore points that are outside the outer AABB "
    "in BOCS\",\"filter\":\"riegl.point_outside_aabb_bocs == "
    "0\",\"activated\":false,\"title\":\"Ignore points outside outer "
    "AABB\"},{\"description\":\"Ignore points that are inside the inner AABB in"
    " BOCS\",\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\",\"activated\":false,\"title\":\"Ignore points inside inner AABB\"}]}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"System "
    "Description\",\"properties\":{\"$class\":{\"description\":\"Object class name\""
    ",\"type\":\"string\",\"enum\":[\"Document\"]},\"author\":{\"description\":\"Name of"
    " the author that created the "
    "document\",\"type\":\"string\"},\"system\":{\"description\":\"The actual system "
    "description, details see documentation of RIEGL System "
    "Description\",\"type\":\"object\"},\"version\":{\"description\":\"Document "
    "format version\",\"type\":\"string\"},\"timestamp\":{\"description\":\"Date and "
    "time of creation (e.g. "
    "2019-06-19T13:32:10+02:00)\",\"type\":\"string\"}},\"description\":\"Details "
    "of major system components like lidar sensors, cameras and navigation "
    "devices\",\"required\":[\"$class\",\"version\",\"author\",\"timestamp\",\"system\"]"
    ",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"$class\":\"Document\",\"author\":\"RIEGL LMS GmbH, "
    "Austria\",\"system\":{\"details\":\"see documentation of RIEGL System Descri"
    "ption\"},\"version\":\"1.2.1\",\"timestamp\":\"2022-09-30T11:56:26+00:00\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Temperature Calculation "
    "Table\",\"properties\":{\"Si\":{\"description\":\"Conversion table for Si chan"
    "nel\",\"$ref\":\"#/definitions/conversion_table\"},\"InGaAs_Si_Difference\":{"
    "\"description\":\"Conversion table for InGaAs - Si difference\",\"$ref\":\"#/"
    "definitions/conversion_table\"},\"InGaAs\":{\"description\":\"Conversion "
    "table for InGaAs channel\",\"$ref\":\"#/definitions/conversion_table\"}},\"d"
    "escription\":\"Conversion of background radiation raw values to "
    "temperatures in \\u00b0C\",\"definitions\":{\"conversion_table\":{\"type\":\"ob"
    "ject\",\"required\":[\"value\",\"temperature\"],\"properties\":{\"temperature\":{"
    "\"description\":\"Temperature [\\u00b0C]\",\"items\":{\"type\":\"number\"},\"type\""
    ":\"array\"},\"value\":{\"description\":\"LSB [1]\",\"items\":{\"type\":\"number\"},\""
    "type\":\"array\"}}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"Si\":{\"temperature\":[546.300048828125,548.8164051212026,551.314393850"
    "0972,554.0144257850053,556.604252334815,559.2124464488079,561.80227299"
    "86177,564.4104671126105,567.0002936624203],\"value\":[0.0,64.00097659230"
    "323,128.0019531846065,192.0029297769097,256.0039063692129,320.00488296"
    "151616,384.0058595538194,448.0068361461226,512.0078127384259]},\"InGaAs"
    "_Si_Difference\":{\"temperature\":[1749.977111117893,1749.977111117893,17"
    "49.977111117893,1749.977111117893,1749.977111117893,1749.977111117893,"
    "1744.7813348796044,1681.9971312601092,1622.3944822534868],\"value\":[100"
    "0.0,1100.090029602954,1200.04425183874,1300.1342814416948,1400.0885036"
    "774805,1500.0427259132668,1600.1327555162209,1700.0869777520065,1800.0"
    "411999877924]},\"InGaAs\":{\"temperature\":[307.22196722535614,309.1153478"
    "247277,311.1188086915047,313.10025350480055,315.2137946389828,317.2172"
    "555057597,319.2207163725366,321.2021611858325,323.3157023200148],\"valu"
    "e\":[0.0,64.00097659230323,128.0019531846065,192.0029297769097,256.0039"
    "063692129,320.00488296151616,384.0058595538194,448.0068361461226,512.0"
    "078127384259]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Time "
    "Base\",\"properties\":{\"epoch\":{\"description\":\"Date and time of timestamp"
    " '0' as proposed by RFC 3339 (e.g. 2015-10-27T00:00:00+01:00).\",\"type\""
    ":\"string\"},\"source\":{\"description\":\"Timestamp source\",\"type\":\"string\","
    "\"enum\":[\"unknown\",\"RTC\",\"GNSS\"]},\"system\":{\"description\":\"Time system "
    "(time standard)\",\"type\":\"string\",\"enum\":[\"unknown\",\"UTC\",\"GPS\"]}},\"des"
    "cription\":\"Base of timestamps (epoch)\",\"required\":[\"epoch\",\"source\"],\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"epoch\":\"2015-10-27T00:00:00+00:00\",\"source\":\"GNSS\",\"system\":\"UTC\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Trajectory Information\",\"properties\":{\"field"
    "_of_application\":{\"description\":\"Field of application\",\"type\":\"string\""
    ",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"]},\""
    "time_interval\":{\"description\":\"Time interval statistics\",\"type\":\"objec"
    "t\",\"required\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"properties\":{"
    "\"average\":{\"description\":\"Average time interval "
    "[s]\",\"type\":\"number\"},\"maximum\":{\"description\":\"Maximum time interval "
    "[s]\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum time interval "
    "[s]\",\"type\":\"number\"},\"std_dev\":{\"description\":\"Standard deviation of "
    "intervals [s]\",\"type\":\"number\"}}},\"location\":{\"description\":\"Project "
    "location, e.g. city/state/country\",\"type\":\"string\"},\"device\":{\"descrip"
    "tion\":\"Navigation device, e.g. "
    "name/type/serial\",\"type\":\"string\"},\"project\":{\"description\":\"Project "
    "name\",\"type\":\"string\"},\"company\":{\"description\":\"Company "
    "name\",\"type\":\"string\"},\"software\":{\"description\":\"Software that "
    "calculated the trajectory (this may be the same or different software "
    "than the one that created the "
    "file)\",\"type\":\"string\"},\"navigation_frame\":{\"description\":\"Navigation "
    "frame (NED: North-East-Down, ENU: East-North-Up)\",\"type\":\"string\",\"enu"
    "m\":[\"unknown\",\"NED\",\"ENU\"]},\"settings\":{\"description\":\"Settings used "
    "to calculate the trajectory (descriptive "
    "text)\",\"type\":\"string\"}},\"description\":\"Details about "
    "position+orientation files\",\"required\":[\"time_interval\",\"navigation_fr"
    "ame\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"field_of_application\":\"MLS\",\"time_interval\":{\"average\":0.00500053,\"m"
    "aximum\":0.005004883,\"minimum\":0.00500032,\"std_dev\":5.51e-07},\"location"
    "\":\"Horn\",\"device\":\"IMU Model 12/1, Serial# 12345\",\"project\":\"Campaign "
    "3\",\"company\":\"RIEGL LMS\",\"software\":\"Navigation Software "
    "XYZ\",\"navigation_frame\":\"NED\",\"settings\":\"default\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"TTIP "
    "Configuration\",\"properties\":{\"dmi_dist_per_tick\":{\"description\":\"dmi, "
    "distance per tick, just informative "
    "[m]\",\"type\":\"number\"},\"dmi_incr\":{\"description\":\"dmi, increment in "
    "ticks\",\"type\":\"integer\"},\"ttip_version\":{\"description\":\"following main"
    ".sub.ss.sss\",\"type\":\"integer\",\"minimum\":0},\"ext_bitmask\":{\"description"
    "\":\"defines which of the internal pulse generators are to be "
    "started\",\"type\":\"integer\",\"minimum\":0},\"input_usage\":{\"description\":\"0"
    " .. deactivated, 1 .. just detecting and timestamping\",\"type\":\"integer"
    "\",\"minimum\":0},\"out_interval\":{\"description\":\"output pulse interval "
    "[0.1 msec]\",\"type\":\"integer\",\"minimum\":0},\"dmi_min_time\":{\"description"
    "\":\"dmi, minimum time interval between trigger events [0.1 "
    "msec]\",\"type\":\"integer\",\"minimum\":0},\"ext_signal\":{\"description\":\"0 .."
    " use input signal, 1 .. use output signal of channel\",\"type\":\"integer\""
    ",\"minimum\":0},\"out_duration\":{\"description\":\"output pulse duration "
    "[0.1 msec]\",\"type\":\"integer\",\"minimum\":0},\"in_max_duration\":{\"descript"
    "ion\":\"stops measurement of pulse duration of input signal [0.1 "
    "msec]\",\"type\":\"integer\",\"minimum\":0},\"channel\":{\"description\":\"ID of "
    "input/output channel\",\"type\":\"integer\",\"minimum\":0},\"ext_delay\":{\"desc"
    "ription\":\"external trigger, time delay [0.1 msec]\",\"type\":\"integer\",\"m"
    "inimum\":0},\"ext_subdivider\":{\"description\":\"reduces the frequency, "
    "default 1\",\"type\":\"integer\",\"minimum\":0},\"out_num_pulses\":{\"descriptio"
    "n\":\"number of output pulses to be generated, 0 .. infinite\",\"type\":\"in"
    "teger\",\"minimum\":0},\"output_descr\":{\"description\":\"descriptive "
    "string\",\"type\":\"string\"},\"in_min_duration\":{\"description\":\"input "
    "signals with smaller pulse durations are ignored [0.1 msec]\",\"type\":\"i"
    "nteger\",\"minimum\":0},\"dmi_max_time\":{\"description\":\"dmi, maximum time "
    "interval between trigger events [0.1 "
    "msec]\",\"type\":\"integer\",\"minimum\":0},\"ext_channel\":{\"description\":\"ID "
    "of channel used as external trigger input, 32 indicates "
    "none\",\"type\":\"integer\",\"minimum\":0},\"in_polarity\":{\"description\":\"0 .."
    " positive edge, 1 .. negative edge\",\"type\":\"integer\",\"minimum\":0},\"num"
    "_channel\":{\"description\":\"number of input/output channels\",\"type\":\"int"
    "eger\",\"minimum\":0},\"out_delay\":{\"description\":\"output pulse initial "
    "delay after start [0.1 "
    "msec]\",\"type\":\"integer\",\"minimum\":0},\"output_usage\":{\"description\":\"0 "
    ".. no output, 1 .. output derived from internal clock, 2 .. output "
    "derived from dmi, 3 .. output derived from external signal, 4 .. "
    "output static low, 5 .. output static high\",\"type\":\"integer\",\"minimum\""
    ":0},\"in_max_delay\":{\"description\":\"maximum delay to output pulse "
    "before fake event is generated [0.1 msec], zero indicates that no fake"
    " events are generated\",\"type\":\"integer\",\"minimum\":0},\"out_polarity\":{\""
    "description\":\"0 .. positive edge, 1 .. negative edge\",\"type\":\"integer\""
    ",\"minimum\":0},\"input_descr\":{\"description\":\"descriptive "
    "string\",\"type\":\"string\"}},\"description\":\"Trigger-Timestamping-Unit/IP "
    "configuration\",\"required\":[\"ttip_version\",\"num_channel\",\"ext_channel\","
    "\"ext_signal\",\"ext_delay\",\"ext_subdivider\",\"ext_bitmask\",\"dmi_incr\",\"dm"
    "i_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\",\"channel\",\"output_descr"
    "\",\"input_descr\",\"output_usage\",\"input_usage\",\"out_polarity\",\"out_durat"
    "ion\",\"out_interval\",\"out_delay\",\"out_num_pulses\",\"in_polarity\",\"in_min"
    "_duration\",\"in_max_duration\",\"in_max_delay\"],\"$schema\":\"http://json-sc"
    "hema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"dmi_dist_per_tick\":0.0,\"dmi_incr\":0,\"ttip_version\":1,\"ext_bitmask\":0"
    ",\"input_usage\":1,\"out_interval\":10000,\"dmi_min_time\":0,\"ext_signal\":0,"
    "\"out_duration\":300,\"in_max_duration\":10000,\"channel\":0,\"ext_delay\":0,\""
    "ext_subdivider\":1,\"out_num_pulses\":1,\"output_descr\":\"Port 1 - Trigger\""
    ",\"in_min_duration\":0,\"dmi_max_time\":0,\"ext_channel\":32,\"in_polarity\":0"
    ",\"num_channel\":9,\"out_delay\":0,\"output_usage\":1,\"in_max_delay\":9990,\"o"
    "ut_polarity\":0,\"input_descr\":\"Port 1 - Exposure\"}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about vertex "
    "file\",\"type\":\"object\",\"title\":\"Vertex Information\",\"properties\":{\"vert"
    "ex_file\":{\"type\":\"object\",\"required\":[\"file_extension\"],\"properties\":{"
    "\"file_uuid\":{\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Vertex file "
    "extension, without the leading dot\",\"type\":\"string\"}}}},\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_uuid\":\"51534d95-d71f-4f36-ae1a-3e63a21fd1c7\",\"fi"
    "le_extension\":\"vtx\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Voxel Information\",\"description\":\"Details "
    "about the voxels contained in the "
    "file\",\"definitions\":{\"shape_thresholds\":{\"description\":\"Thresholds "
    "used to compute the voxel's shape_id value.\",\"type\":\"object\",\"required"
    "\":[\"plane\",\"line\"],\"properties\":{\"line\":{\"description\":\"If the biggest"
    " eigenvalue is bigger than the median eigenvalue * line_threshold, the"
    " voxel is considered a line.\",\"exclusiveMinimum\":true,\"type\":\"number\","
    "\"minimum\":1},\"plane\":{\"type\":\"number\",\"minimum\":0,\"description\":\"If "
    "the smallest eigenvalue is smaller than the median eigenvalue * "
    "plane_threshold, the voxel is considered a plane.\",\"exclusiveMinimum\":"
    "true,\"exclusiveMaximum\":true,\"maximum\":1}}},\"voxel_type\":{\"description"
    "\":\"Whether a point in a voxel represents its center or its centroid. "
    "If type is <tt>index</tt> there is no point but only an integer voxel "
    "index.\",\"enum\":[\"center\",\"centroid\",\"index\"],\"default\":\"centroid\"},\"vo"
    "xel_size\":{\"items\":{\"$ref\":\"#/definitions/edge_length\"},\"description\":"
    "\"Size of voxels.\",\"type\":\"array\",\"maxItems\":3,\"minItems\":3},\"voxel_ori"
    "gin_point\":{\"description\":\"Origin point for all voxel indices in voxel"
    " CRS.\",\"items\":{\"type\":\"number\"},\"type\":\"array\",\"maxItems\":3,\"minItems"
    "\":3},\"voxel_origin_enum\":{\"description\":\"Defines whether the voxel's "
    "center or a corner is placed on CRS origin <tt>(0/0/0)</tt>.\",\"enum\":["
    "\"center\",\"corner\"],\"default\":\"corner\"},\"voxel_size_cubic\":{\"type\":\"num"
    "ber\",\"$ref\":\"#/definitions/edge_length\"},\"edge_length\":{\"description\":"
    "\"Length of voxel edge [m].\",\"exclusiveMinimum\":true,\"type\":\"number\",\"m"
    "inimum\":0}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"oneOf"
    "\":[{\"additionalProperties\":false,\"required\":[\"size\",\"voxel_origin\",\"vo"
    "xel_type\"],\"properties\":{\"shape_thresholds\":{\"$ref\":\"#/definitions/sha"
    "pe_thresholds\"},\"voxel_origin\":{\"$ref\":\"#/definitions/voxel_origin_enu"
    "m\"},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_type\"},\"size\":{\"descript"
    "ion\":\"Size of voxels in file coordinate system.\",\"oneOf\":[{\"$ref\":\"#/d"
    "efinitions/voxel_size\"},{\"$ref\":\"#/definitions/voxel_size_cubic\"}]}}},"
    "{\"additionalProperties\":false,\"required\":[\"reference_point\",\"size_llcs"
    "\",\"size\",\"voxel_origin\",\"voxel_type\"],\"properties\":{\"shape_thresholds\""
    ":{\"$ref\":\"#/definitions/shape_thresholds\"},\"voxel_origin\":{\"oneOf\":[{\""
    "$ref\":\"#/definitions/voxel_origin_enum\"},{\"description\":\"The base "
    "point of the voxel grid. Used together with <tt>voxel_size</tt> and "
    "<tt>voxel_index</tt> to compute actual point coordinates.\",\"$ref\":\"#/d"
    "efinitions/voxel_origin_point\"}]},\"size_llcs\":{\"description\":\"Size of "
    "voxels in a locally levelled cartesian coordinate system (xyz). This "
    "is only used for voxels based on a map projection.\",\"$ref\":\"#/definiti"
    "ons/voxel_size\"},\"reference_point\":{\"items\":{\"type\":\"number\",\"maximum\""
    ":180,\"minimum\":-180},\"description\":\"Point in WGS84 geodetic decimal "
    "degree (EPSG:4326) that was used to compute the projection distortion "
    "parameters. The coefficient order is latitude, longitude. Only voxels "
    "with corresponding geo_tag, voxel_size and reference_point can be "
    "reliably processed together. This entry is available for voxel files "
    "in projected CRS only.\",\"type\":\"array\",\"maxItems\":2,\"minItems\":2},\"vox"
    "el_type\":{\"$ref\":\"#/definitions/voxel_type\"},\"size\":{\"description\":\"Si"
    "ze of voxels in file coordinate "
    "system.\",\"$ref\":\"#/definitions/voxel_size\"}}}]}"
    ,
    /* example: */
    "{\"shape_thresholds\":{\"line\":6,\"plane\":0.16},\"voxel_origin\":\"corner\",\"s"
    "ize_llcs\":[0.5156575252891171,0.5130835356683303,0.5143705304787237],\""
    "reference_point\":[48,16],\"voxel_type\":\"centroid\",\"size\":[0.59716428347"
    "79395,0.5971642834779395,0.5143705304787237]}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Waveform Averaging "
    "Settings\",\"properties\":{\"num_shots\":{\"description\":\"Number of "
    "consecutive shots to be used for averaging.\",\"type\":\"integer\",\"minimum"
    "\":1},\"mta_zone\":{\"description\":\"Fixed MTA zone for averaging.\",\"type\":"
    "\"integer\",\"minimum\":1},\"trim\":{\"description\":\"Percentage for robust av"
    "eraging.\",\"type\":\"number\",\"maximum\":0.5,\"minimum\":0,\"default\":0}},\"des"
    "cription\":\"Settings for waveform averaging\",\"required\":[\"num_shots\",\"m"
    "ta_zone\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"num_shots\":7,\"mta_zone\":1,\"trim\":0.05}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Waveform Information\",\"properties\":{\"sample_"
    "data_files\":{\"items\":{\"type\":\"object\",\"required\":[\"channel\",\"channel_n"
    "ame\",\"sample_interval\",\"sample_bits\",\"laser_wavelength\",\"delta_st\",\"fi"
    "le_extension\"],\"properties\":{\"channel_name\":{\"description\":\"Sample "
    "block channel "
    "name\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Sample data "
    "file extension, without the leading "
    "dot\",\"type\":\"string\"},\"laser_wavelength\":{\"description\":\"Laser "
    "wavelength in meters (0 = unknown)\",\"exclusiveMinimum\":false,\"type\":\"n"
    "umber\",\"minimum\":0},\"channel\":{\"type\":\"integer\",\"minimum\":0,\"descripti"
    "on\":\"Sample block channel number (255 = invalid)\",\"exclusiveMinimum\":f"
    "alse,\"exclusiveMaximum\":false,\"maximum\":255},\"delta_st\":{\"description\""
    ":\"reserved\",\"type\":\"number\"},\"sample_interval\":{\"description\":\"Samplin"
    "g interval in seconds\",\"exclusiveMinimum\":false,\"type\":\"number\",\"minim"
    "um\":0},\"sample_bits\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"Bitw"
    "idth of samples (e.g. 10 bit, 12 bit)\",\"exclusiveMinimum\":false,\"exclu"
    "siveMaximum\":false,\"maximum\":32}}},\"type\":\"array\"},\"sample_block_file\""
    ":{\"type\":\"object\",\"required\":[\"file_extension\"],\"properties\":{\"file_uu"
    "id\":{\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Sample block "
    "file extension, without the leading dot\",\"type\":\"string\"}}},\"range_off"
    "set_waveform_samples_m\":{\"description\":\"Calibrated device specific "
    "range offset for waveform samples in "
    "meters.\",\"type\":\"number\"},\"range_offset_m\":{\"description\":\"Calibrated "
    "device specific range offset for waveform analysis by system response "
    "fitting in meters.\",\"type\":\"number\"}},\"description\":\"Details about "
    "waveform files\",\"required\":[\"sample_block_file\",\"sample_data_files\"],\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"sample_data_files\":[{\"channel_name\":\"high_power\",\"file_uuid\":\"da0844"
    "13-e3e8-4655-a122-071de8490d8e\",\"channel\":0,\"laser_wavelength\":0,\"file"
    "_extension\":\"sp0\",\"delta_st\":0,\"sample_interval\":1.00503e-09,\"sample_b"
    "its\":12},{\"channel_name\":\"low_power\",\"file_uuid\":\"93585b5e-5ea9-43a1-9"
    "47b-e7ba3be642d2\",\"channel\":1,\"laser_wavelength\":0,\"file_extension\":\"s"
    "p1\",\"delta_st\":0,\"sample_interval\":1.00503e-09,\"sample_bits\":12},{\"cha"
    "nnel_name\":\"wwf\",\"file_uuid\":\"9d2298c4-1036-464f-b5cb-1cf8e517f3a0\",\"c"
    "hannel\":5,\"laser_wavelength\":0,\"file_extension\":\"sp5\",\"delta_st\":0,\"sa"
    "mple_interval\":1.00503e-09,\"sample_bits\":12}],\"sample_block_file\":{\"fi"
    "le_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\",\"file_extension\":\"sbx\""
    "},\"range_offset_waveform_samples_m \":7.283,\"range_offset_m\":3.1415}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scanner settings for waveform output\",\"items\":{\"type\":"
    "\"object\",\"required\":[\"sbl_name\",\"enabled\",\"channel_idx_mask\"],\"propert"
    "ies\":{\"smart_enabled\":{\"description\":\"Smart waveform output enabled\",\""
    "type\":\"boolean\"},\"pass_rng_greater\":{\"description\":\"Threshold for "
    "range greater than "
    "[m]\",\"type\":\"number\"},\"pass_dev_greater\":{\"description\":\"Threshold for"
    " deviation greater than "
    "[1]\",\"type\":\"integer\"},\"enabled\":{\"description\":\"Waveform output enabl"
    "ed\",\"type\":\"boolean\"},\"pass_ampl_greater\":{\"description\":\"Threshold "
    "for amplitude greater than "
    "[dB]\",\"type\":\"number\"},\"pass_dev_less\":{\"description\":\"Threshold for "
    "deviation less than "
    "[1]\",\"type\":\"integer\"},\"pass_rng_less\":{\"description\":\"Threshold for "
    "range less than "
    "[m]\",\"type\":\"number\"},\"logic_expression\":{\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\",\"type\":\"string\"},\"sbl_name\":{\"description\":\"Name of sample "
    "block, e.g.: wfm, "
    "wwf\",\"type\":\"string\"},\"pass_ampl_less\":{\"description\":\"Threshold for "
    "amplitude less than "
    "[dB]\",\"type\":\"number\"},\"channel_idx_mask\":{\"description\":\"Bit mask for"
    " channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, ...\",\"type\":\"integer\"}}},\"type\":\"array\",\"title\":\"Waveform "
    "Settings\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "[{\"smart_enabled\":true,\"pass_rng_greater\":9.27,\"enabled\":true,\"pass_am"
    "pl_greater\":1.0,\"pass_rng_less\":13.11,\"sbl_name\":\"wfm\",\"pass_ampl_less"
    "\":5.0,\"channel_idx_mask\":11},{\"enabled\":false,\"sbl_name\":\"wwf\",\"channe"
    "l_idx_mask\":32}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Window Analysis\",\"properties\":{\"filter\":{\"ty"
    "pe\":\"object\",\"required\":[\"angle\",\"range_min\",\"range_max\",\"amplitude_ma"
    "x\"],\"properties\":{\"range_min\":{\"items\":{\"type\":\"number\"},\"description\""
    ":\"[m]\",\"type\":\"array\"},\"range_max\":{\"items\":{\"type\":\"number\"},\"descrip"
    "tion\":\"[m]\",\"type\":\"array\"},\"amplitude_max\":{\"items\":{\"type\":\"number\"}"
    ",\"description\":\"[dB]\",\"type\":\"array\"},\"angle\":{\"items\":{\"type\":\"number"
    "\"},\"description\":\"[deg]\",\"type\":\"array\"}}},\"result\":{\"type\":\"object\",\""
    "required\":[\"angle\",\"range_mean\",\"range_sigma\",\"amplitude_mean\",\"amplit"
    "ude_sigma\",\"amplitude_offset\"],\"properties\":{\"amplitude_offset\":{\"item"
    "s\":{\"type\":\"number\"},\"description\":\"[dB]\",\"type\":\"array\"},\"range_sigma"
    "\":{\"items\":{\"type\":\"number\"},\"description\":\"[m]\",\"type\":\"array\"},\"ampl"
    "itude_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"[dB]\",\"type\":\"ar"
    "ray\"},\"angle\":{\"items\":{\"type\":\"number\"},\"description\":\"[deg]\",\"type\":"
    "\"array\"},\"timestamp\":{\"items\":{\"type\":\"number\"},\"description\":\"[s]\",\"t"
    "ype\":\"array\"},\"range_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"["
    "m]\",\"type\":\"array\"},\"amplitude_sigma\":{\"items\":{\"type\":\"number\"},\"desc"
    "ription\":\"[dB]\",\"type\":\"array\"}}},\"settings\":{\"type\":\"object\",\"require"
    "d\":[\"range\",\"amplitude\"],\"properties\":{\"amplitude\":{\"type\":\"object\",\"r"
    "equired\":[\"sigma_factor\",\"additive_value\"],\"properties\":{\"sigma_factor"
    "\":{\"type\":\"number\"},\"additive_value\":{\"type\":\"number\"}}},\"range\":{\"typ"
    "e\":\"object\",\"required\":[\"sigma_factor\",\"additive_value\"],\"properties\":"
    "{\"sigma_factor\":{\"type\":\"number\"},\"additive_value\":{\"type\":\"number\"}}}"
    "}}},\"description\":\"Window analysis data estimated from scandata and "
    "resulting filter parameters\",\"required\":[\"result\",\"filter\",\"settings\"]"
    ",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"filter\":{\"range_min\":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.218,-0.21"
    "9,-0.221,-0.223,-0.225,-0.227],\"range_max\":[0.424,0.425,0.426,0.427,0."
    "428,0.428,0.429,0.43,0.431,0.431,0.432],\"amplitude_max\":[8.04,8.01,7.9"
    "9,7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78],\"angle\":[14.9,15.0,15.1,15.2,"
    "15.3,15.4,15.5,15.6,15.7,15.8,15.9]},\"result\":{\"amplitude_offset\":[1.9"
    ",1.9],\"range_sigma\":[0.01869652,0.02151435,0.01747969,0.01918765,0.019"
    "45776,0.01934862,0.01955329,0.02225589,0.02229977,0.01899122,0.0200943"
    "3],\"amplitude_mean\":[5.347396,5.263155,5.224655,5.179926,5.097782,5.11"
    "6479,5.051756,4.983473,5.007885,5.002441,4.982],\"angle\":[14.9,15.0,15."
    "1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],\"timestamp\":[408.4441,411.4"
    "443],\"range_mean\":[0.1105621,0.1079564,0.1087088,0.1067261,0.1054582,0"
    ".1090412,0.102871,0.1019044,0.1051523,0.1058445,0.1031261],\"amplitude_"
    "sigma\":[0.4272844,0.4298479,0.4236816,0.4283583,0.4362353,0.4315141,0."
    "4373984,0.4472798,0.4346001,0.4345487,0.4540681]},\"settings\":{\"amplitu"
    "de\":{\"sigma_factor\":4,\"additive_value\":1.0},\"range\":{\"sigma_factor\":8,"
    "\"additive_value\":0.1}}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Window Echo "
    "Correction\",\"properties\":{\"amplitude\":{\"description\":\"Amplitude axis "
    "of correction table\",\"type\":\"object\",\"required\":[\"minimum\",\"maximum\",\""
    "entries\"],\"properties\":{\"maximum\":{\"description\":\"Maximum amplitude in"
    " dB\",\"type\":\"number\",\"minimum\":0.0},\"minimum\":{\"description\":\"Minimum "
    "amplitude in "
    "dB\",\"type\":\"number\",\"minimum\":0.0},\"entries\":{\"description\":\"Number of"
    " amplitude "
    "entries\",\"type\":\"integer\",\"minimum\":1}}},\"range\":{\"description\":\"Range"
    " axis of correction table\",\"type\":\"object\",\"required\":[\"minimum\",\"maxi"
    "mum\",\"entries\"],\"properties\":{\"maximum\":{\"description\":\"Maximum range "
    "in m\",\"type\":\"number\",\"maximum\":2.0,\"minimum\":-2.0},\"minimum\":{\"descri"
    "ption\":\"Minimum range in m\",\"type\":\"number\",\"maximum\":2.0,\"minimum\":-2"
    ".0},\"entries\":{\"description\":\"Number of range entries\",\"type\":\"integer"
    "\",\"minimum\":1}}},\"slices\":{\"items\":{\"description\":\"Window echo "
    "correction parameter slice\",\"type\":\"object\",\"required\":[\"amplitude\",\"t"
    "able\"],\"properties\":{\"amplitude\":{\"description\":\"Window echo amplitude"
    " of slice in dB\",\"type\":\"number\"},\"table\":{\"description\":\"Correction "
    "table (dimension defined by the 'amplitude' and 'range' "
    "objects)\",\"items\":{\"description\":\"Table row (= amplitude "
    "axis)\",\"items\":{\"description\":\"Table column (= range "
    "axis)\",\"items\":{\"description\":\"Table cell (item 0: amplitude in dB, 1:"
    " range in m, 2: flags)\",\"type\":\"number\"},\"type\":\"array\",\"maxItems\":3,\""
    "minItems\":3},\"type\":\"array\",\"minItems\":1},\"type\":\"array\",\"minItems\":1}"
    "}},\"type\":\"array\"}},\"description\":\"Correction parameters for window "
    "glass echoes\",\"required\":[\"amplitude\",\"range\",\"slices\"],\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude\":{\"maximum\":20,\"minimum\":2,\"entries\":128},\"range\":{\"maximu"
    "m\":1.5060822940732335,\"minimum\":-1.5060822940732335,\"entries\":128},\"sl"
    "ices\":[{\"amplitude\":1.5,\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]"
    "},{\"amplitude\":2.0,\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]}]}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const std::size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);
const std::vector<const MetaDataItemInfo*> RDB_META_DATA_ITEMS_VECTOR(
    RDB_META_DATA_ITEMS, RDB_META_DATA_ITEMS + RDB_META_DATA_ITEM_COUNT
);

}}} // namespace riegl::rdb::pointcloud

#endif // RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
